;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.area _DATA
.area _CODE

.include "cpctglobals.h.s"
.include "entities/entity.h.s"
.include "renderutils.h.s"
.include "macros/cpct_undocumentedOpcodes.h.s"
.include "globals.h.s"
.include "gameplaystatics.h.s"

;; Utils class composed by "static" functions that implement
;; basic rendering functionalities


;;=============================================
;;=============================================
;; PRIVATE DATA
;;=============================================
;;=============================================
;;
;; Video memory pointers
;;
.equ    VMEM_0,      0xC0
.equ    VMEM_1,      0x80             ;; TBD: Erase if we don't need double buffer
.equ    VMEM_SIZE,   0x4000
current_vm:     .db VMEM_0              ;; Current video memory
back_vm:        .db VMEM_1              ;; Back buffer memory pointer

;;=============================================
;; Switches video buffer
;;=============================================
ru_switchBuffer::
	ld a, (back_vm)
	ld b, a
	ld a, (current_vm)
	ld (back_vm), a
	ld a, b
	ld (current_vm), a

	srl b
	srl b
	ld l, b
	jp cpct_setVideoMemoryPage_asm

ru_initBackBuffer::
    ld hl, #0x8000
    ld (hl), #0
    ld de, #0x8001
    ld bc, #0x3999
    ldir
    
    ret
;;=============================================
;; Returns the location of the current video memory
;; DESTROYS: DE
;; RETURNS: DE => Current_vm
;;=============================================
ru_getCurrentVM::
    ld a, (current_vm)
    ld d, a
    ld e, #0
    ret

;;=============================================
;; Returns the location of the back buffer video memory
;; DESTROYS: DE
;; RETURNS: DE => Current_vm
;;=============================================
ru_getBackVM::
    ld a, (back_vm)
    ld d, a
    ld e, #0
    ret

;;=============================================
;; Returns the location of the back buffer video memory
;; DESTROYS: DE
;; RETURNS: DE => Current_vm
;;=============================================
ru_getVmOffset::
    ld a, (back_vm)
    ld d, a
    ld e, #MAP_OFFSET_Y
    ret


;;====================================================================================
;; Draws a masked entity
;; 
;; INPUTS:
;;      IY => Pointer to entity data
;;      Inputs for cpct_drawSpriteMaskedAlignedTable_asm: 
;;          (2B BC) psprite	Source Sprite Pointer
;;          (2B DE) pvideomem	Destination video memory pointer
;;          (1B IXL) width	Sprite Width in bytes (>0) (Beware, not in pixels!)
;;          (1B IXH) height	Sprite Height in bytes (>0)
;;          (2B HL) pmasktable	Pointer to the aligned mask table used to create transparency
;; DESTROYS:
;;      DE, BC, HL, AF, IX, IY
;;====================================================================================   
ru_drawMaskedEntity::

    ld d, Ent_h(iy)                        ;; |
    ld e, Ent_w(iy)                        ;; | Temporally store entity height and width on DE
        
    .db 0xDD                               ;;|
    .db 0x62                               ;;| ld ixl, d                   
    
    .db 0xDD                               ;;|
    .db 0x6B                               ;;| ld ixh, e                   
        
    ;; Calculate Screen position
    ld   c, Ent_x(iy)                      ;; C=Entity_X
    ld   b, Ent_y(iy)                      ;; B=Entity_Y
    
    call ru_getBackVM
    call cpct_getScreenPtr_asm             ;; Get Pointer to Screen
    ex de, hl    
    
    ld b, Ent_spr_h(iy)                    ;;|
    ld c, Ent_spr_l(iy)                    ;;| BC points to entity sprite
    ld hl, #trans_table_loc                 ;; HL points to our aligned mask table

    jp cpct_drawSpriteMaskedAlignedTable_asm

;;===================================================================
;; Returns in DE instead of HL the video memory pointer
;; Input:
;;     (1B C ) X position
;;     (1B B ) Y position
;; Returns:
;;      DE => Mem screen ptr
;; DESTROYS: AF, BC, DE, HL
;;===================================================================
ru_gretScreenPointerOnDEBasedOnPosition::
    call ru_getCurrentVM
    call cpct_getScreenPtr_asm             ;; Get Pointer to Screen
    ex de, hl
    ret


;;===================================================================
;; Input:
;;     (D) Background color
;;     (E) Foreground color
;;     (BC) Location on the screen B: x, C:y
;;     (IY) Pointer to the start of the string
;; Returns:
;;      DE => Mem screen ptr
;; DESTROYS: AF, BC, DE, HL
;; WARNING: Do not put this function’s code below 0x4000 in memory
;;===================================================================
ru_drawText::
    push bc
    call cpct_setDrawCharM0_asm   ;; Set draw char colours
    pop bc
    call ru_getCurrentVM
    call cpct_getScreenPtr_asm
    jp cpct_drawStringM0_asm


;;===================================================================
;; Draws a cute context menu frame, thanks to Azicuetano for the pattern.
;; DESTROYS: AF, BC, DE, HL
;;===================================================================
ru_drawContextMenuFrame::
    call cpct_waitVSYNC_asm
    ;; White outer 1 up
    ld bc, #0x4613
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xff
    ld bc, #0x0228
    call cpct_drawSolidBox_asm
    
    ;; White outer 2 down
    ld bc, #0x7E13
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xff
    ld bc, #0x0228
    call cpct_drawSolidBox_asm
    
    ;; Dark blue touch 1 up
    ld bc, #0x4813
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xc0
    ld bc, #0x0228
    call cpct_drawSolidBox_asm
    
    ;; Dark blue touch 2 down 
    ld bc, #0x7C13
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xc0
    ld bc, #0x0228
    call cpct_drawSolidBox_asm

    ;; Big black box center
    ld bc, #0x4A13
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0x00
    ld bc, #0x3228
    call cpct_drawSolidBox_asm

    ;; Thin side left
    ld bc, #0x4A12
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xEA
    ld bc, #0x3201
    call cpct_drawSolidBox_asm
    
    ;; Thin detail up right
    ld bc, #0x483B
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xEA
    ld bc, #0x0201
    call cpct_drawSolidBox_asm
    
    ;; Thin detail bottom right
    ld bc, #0x7C3B
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xEA
    ld bc, #0x0201
    call cpct_drawSolidBox_asm

    ;; Thin side right
    ld bc, #0x4A3B
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xD5
    ld bc, #0x3201
    call cpct_drawSolidBox_asm

    ;; Thin detail up left
    ld bc, #0x4812
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xD5
    ld bc, #0x0201
    call cpct_drawSolidBox_asm

    ;; Thin detail bottom left
    ld bc, #0x7C12
    call ru_gretScreenPointerOnDEBasedOnPosition
    ld a, #0xD5
    ld bc, #0x0201
    jp cpct_drawSolidBox_asm


;;===================================================================
;;  INPUTS:
;;       D: color
;;===================================================================
ru_drawFullScreenWithColor:
    ld hl, #draw_full_screen_color_1
    ld (hl), d
    ld hl, #draw_full_screen_color_2
    ld (hl), d

    ld de, #0xC000
    draw_full_screen_color_1 = . + 1
    ld  a, #WHITE_COLOR_M1
    ld bc, #0x3FFF
    call cpct_memset_asm
    ld de, #0x8000
    draw_full_screen_color_2 = . + 1
    ld  a, #WHITE_COLOR_M1
    ld bc, #0x3FFF
    jp cpct_memset_asm

;;===================================================================
;;  Fades out the screen
;;===================================================================
ru_fadeOut::
    ld d, #WHITE_COLOR_M1
    call ru_drawFullScreenWithColor
    ld d, #NVSYNCS_FADE_IN_OUT + 5
    call waitNVSYNCs

    ld d, #GREY_COLOR_M1
    call ru_drawFullScreenWithColor
    ld d, #NVSYNCS_FADE_IN_OUT
    call waitNVSYNCs

    ld d, #BLACK_COLOR_M1
    call ru_drawFullScreenWithColor
    ld d, #NVSYNCS_FADE_IN_OUT
    jp waitNVSYNCs

;;===================================================================
;;  Fades in the screen
;;===================================================================
ru_fadeIn::
    ld d, #BLACK_COLOR_M1
    call ru_drawFullScreenWithColor
    ld d, #NVSYNCS_FADE_IN_OUT + 5
    call waitNVSYNCs

    ld d, #GREY_COLOR_M1
    call ru_drawFullScreenWithColor
    ld d, #NVSYNCS_FADE_IN_OUT
    call waitNVSYNCs

    ld d, #WHITE_COLOR_M1
    call ru_drawFullScreenWithColor
    ld d, #NVSYNCS_FADE_IN_OUT
    jp waitNVSYNCs