;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.area _CODE

.include "cpctglobals.h.s"
.include "entities/entity.h.s"
.include "macros/cpct_maths.h.s"

;; Utils class composed by "static" functions that implement
;; basic functionalities such as collision checks or primitives
;; needed for ease of use

;;===============================================================
;;===============================================================
;; PUBLIC FUNCTIONS 
;;===============================================================
;;===============================================================

;;===============================================================
;; Checks Collision between entity A and entity B
;; Input:
;;      HL => Points to entity A
;;      IY => Points to entity B        
;;  Main requirements:
;;      The data will be analysed in this order:
;;          - pos_x
;;          - pos_y
;;          - width
;;          - height
;;  RETURNS:
;;      A: #0xFF == collision || 00 == no_collision  
;;      TBD: Not sure if 00 and x0xFF is ideal to return
;;  DESTROYS: AF, BC
;;===============================================================
isColliding::
    ;;
    ;; if ((ent_A_x + ent_A_w) <= ent_B_x) no_collision
    ;;  
    ;; (ent_A_x + ent_A_w) - ent_B_x <= 0  
    ld      a, (hl)             ;; | A = ent_A_x   
    inc     hl                  ;; | 
    inc     hl                  ;; | increment 2 to get the width
    add     (hl)                ;; Result (ent_A_x + ent_A_w)
    dec     hl                  ;; |
    dec     hl                  ;; | Restoring hl to defaults
    ld      c, Ent_x(iy)        ;; storing in c ent_B_x         
    sub     c                   ;; (ent_A_x + ent_A_w) - ent_B_x
    jr      z, no_collision     ;; if( <= 0 )
    jp      m, no_collision     ;;  then no_collision

    ;;
    ;; if((ent_B_x + ent_B_w) <= ent_A_x)
    ;;
    ;; (ent_B_x + ent_B_w) - ent_A_x <= 0 
    ld      b, Ent_x(iy)        ;; storing in b ent_B_x  
    ld      a, Ent_w(iy)        ;; storing in a ent_B_w
    add     b                   ;; A = (ent_A_x + ent_A_w)
    sub     (hl)                ;; (ent_B_x + ent_B_w) - ent_A_x
    jr      z, no_collision     ;; if( <= 0 )
    jp      m, no_collision     ;;  then no_collision

    ;;
    ;; if ((ent_A_y + ent_A_h) <= ent_B_y) no_collision
    ;;  
    ;; (ent_A_y + ent_A_h) - ent_B_y <= 0  
    inc     hl                  ;; |
    ld      a, (hl)             ;; | A = ent_A_y   
    inc     hl                  ;; | 
    inc     hl                  ;; | increment 2 to get the height
    add     (hl)                ;; Result (ent_A_y + ent_A_h)
    dec     hl                  ;; |
    dec     hl                  ;; |
    dec     hl                  ;; | Restoring hl to defaults
    ld      c, Ent_y(iy)        ;; storing in a ent_B_y     
    sub     c                   ;; (ent_A_y + ent_A_h) - ent_B_y
    jr      z, no_collision     ;; if( <= 0 )
    jp      m, no_collision     ;;  then no_collision

    ;;
    ;; if((ent_B_y + ent_B_h) <= ent_A_y)
    ;;
    ;; (ent_B_y + ent_B_h) - ent_A_y <= 0 
    ld      b, Ent_y(iy)        ;; | B = ent_B_y
    ld      a, Ent_h(iy)        ;; storing in a ent_B_h
    add     b                   ;; A = (ent_A_y + ent_A_h)
    inc     hl                  ;; HL = ent_A_y
    sub     (hl)                ;; (ent_B_y + ent_B_h) - ent_A_y
    dec     hl                  ;; restoring hl to defaults
    jr      z, no_collision     ;; if( <= 0 )
    jp      m, no_collision     ;;  then no_collision
 

    ;; Collision
    ld      a, #0xFF
    ret
    ;; No collision
    no_collision:
        ld      a, #00
    ret


;;===============================================================
;; Divides D by E 
;; INPUT:
;;      D => Dividend
;;      E => Divisor
;; RETURNS:
;;      D => Quotient
;;      A => Remainder
;; DESTROYS: A, DE, BC
;;===============================================================
div8::

    div_d_e:
        xor	a
        ld	b, #08

    _loop:
        sla	d
        rla
        cp	e
        jr	c, .+4
        sub	e
        inc	d

        djnz _loop

        ret


;;======================================
;; Input:
;;    C => factor A 
;;    D  => factor B
;; Returns:
;;    A => Mult result
;; Destroys: B
;;======================================
mul8::
    ld b, #8          ;7           7
    xor a           ;4           4
        rlca          ;4*8        32
        rlc c         ;8*8        64
        jr nc,.+3     ;(12|11)    96|88
        add a,d     ;--
        djnz .-6      ;13*7+8     99
    ret             ;10         10


;;======================================
;; Input:
;;    DE => factor A 
;;    A  => factor B
;; Returns:
;;    HL => Mult result
;; Destroys: B, HL
;; NOTE: multiplication UNSIGNED
;;======================================
mul16::
    ld  b, #8           ;7           7
    ld hl, #0           ;10         10
        add hl, hl      ;11*8       88
        rlca            ;4*8        32
        jr nc, .+3      ;(12|18)*8  96+6x
            add hl, de  ;--         --
        djnz .-5        ;13*7+8     99
    ret                 ;10         10


;;===============================================================
;; Cycles throught the sprites of a sprite array in order
;; jumping 2 by 2 due to word length
;; Input:
;;      A  => Num of cycles to loop through in this format: 2 * num_anim_frames
;;      IY => Points to the pawn to animate        
;;      HL => Points to the begining of the anim table
;;  DESTROYS: HL, IY, DE, BC
;;===============================================================
setLoopableAnimation::
    inc Pawn_frame(iy)        ;; Calculating next pawn frame
    ld b, Pawn_frame(iy)      ;; Loading in e d+1 pawn frame

    ;; This loop resolves the anim ptr
    resolveptr_loop:
        ld d, (hl)
        inc hl
        ld e, (hl)
    djnz resolveptr_loop 

    ld Ent_spr_l(iy), d     ;; |
    ld Ent_spr_h(iy), e     ;; | Refreshing the new sprite
    inc Pawn_frame(iy)      ;; Post increment pawn frame for the next iteration
    cp Pawn_frame(iy)       ;; CurrentFrame != anim_cycles?
    ret nz                  ;; Then -> ret and wait next it

    ;; We reset to 0
    ld Pawn_frame(iy), #0   ;; Reset back values to 0

    ret

;;======================================
;; It multiplies all values of a table
;;   by -1 and finish when finds 0x80
;; Input:
;;    HL => Ptr to table
;; Returns:
;; Destroys: HL, A
;;======================================
mulNeg1Array::
    ld  a, (hl)
    cp  #0x80
    ret z
    xor #0xFF
    add #1
    ld  (hl), a
    inc hl
    jr mulNeg1Array

;;======================================
;;  It multiplies a 16 bit register
;;  by -1
;;  INPUTS:
;;      HL: register to be multiplied
;;  OUTPUTS:
;;      HL: register with result
;;  DESTROYS: AF, HL
;;======================================
mulNeg1Reg16::
    xor a       ;;4
	sub l       ;;4
	ld l,a      ;;4
	sbc a,a     ;;4
	sub h       ;;4
	ld h,a      ;;4 
	ret         ;;24 cycles


;;======================================
;;  It returns the abs value
;;  INPUTS:
;;       A: positive or negative number
;;  OUTPUTS:
;;       A: positive number
;;  DESTROYS: 
;;======================================
absValue::
    bit 7,a
    ret z
    neg
    ret

;;absValueOld::
;;    sla a 
;;    jr c, absValueOfNegValue
;;    sra a
;;    res 7,a
;;    ret
;;
;;absValueOfNegValue:
;;    sra a
;;    xor #0b11111111
;;    res 7,a
;;    inc a
;;    ret

;;======================================
;;  It checks if a number is greater
;;  or equal than other
;;  INPUTS:
;;       H: first number
;;       L: second number
;;  OUTPUTS:
;;       A: 0x00 if H < L
;;          0x01 if H >= L
;;  DESTROYS: AF
;;======================================
greaterOrEqual::
    ld a, h         ;; A = H = first number
    sub l           ;; A -= L = first number - second number
    ld a, #0        ;; A = 0
    ret m           ;; Return A = 0 if result is negative (i.e. first number - second number < 0 not >= 0)
    inc a           ;; A = 1
    ret             ;; Return A = 1 because above condition ('ret m') is false so H >= L

;;======================================
;;  It checks if a number is between
;;  a range of 2 numbers
;;  INPUTS:
;;       H: first range number
;;       L: second range number
;;       B: number to check
;;  NOTE: H have to be less or equal than L
;;  OUTPUTS:
;;       A: 0x00 if H > B or L > B
;;          0x01 if H <= B <= L
;;  DESTROYS: AF
;;======================================
checkNumberBetweenRange::
    ld a, l                         ;; A = L = second range number
    sub b                           ;; A = L - B = second range number - number to check
    jr nc, continueCheckingRange    ;; If no negative number (i.e. >= 0), continue checking
        xor a                    ;; A = 0 because  L - B is negative so L > B
        ret

    continueCheckingRange:
    ld a, b
    sub h
    jr nc, endCheckingRange
        xor a
        ret
    
    endCheckingRange:
    ld a, #1
    ret


;;============================================================================
;; Converts a numeric value onto a string
;; INPUTS: 
;;  HL => Number to convert (hl < 65556) 
;;  DE => Pointer to the ascii string (length 3, example: "000")
;;  Returns:
;;  DE => Updated string
;;============================================================================
Num2Dec8b::
  ld  bc, #-100
  call  Num1b
  ld  c, #-10
  call  Num1b
  ld  c,b
Num1b:
 ld  a,#'0'-1
Num2b:
  inc a
  add hl,bc
  jr  c,Num2b
  sbc hl,bc

  ld  (de),a
  inc de
  ret


;;============================================================================
;; Waits n vsyncs
;; INPUTS: 
;;  D => Number of halts to wait
;; DESTROYS: DE, AF, BC
;;============================================================================
waitNVSYNCs::
    init_gs_vsync_loop:
        call cpct_waitVSYNC_asm
        dec d
        ret z
        halt
        halt  
    jr nz, init_gs_vsync_loop
    ret

;;==============================================================
;; Returns the key id of the last key pressed
;; This function was created by Ronaldo on Amstrad.es
;; DESTROYS: AF, DE
;; OUTPUT: 
;;      HL => cpct_keyID
;;       A => 00 no keys where pressed | 01 keys where pressed
;;==============================================================
.globl _cpct_keyboardStatusBuffer
waitForKey::
    ld hl, #_cpct_keyboardStatusBuffer
    ld d, #0
    ld e, #9
    add hl, de

    ;; Wait until a key is pressed
    push hl                       ;; since hl and de get destroyed scanning the keyboard we use the stack
    wfk_wait_for_key:
        ;; tracks if any key is pressed
        call cpct_scanKeyboard_asm
        call cpct_isAnyKeyPressed_asm  
    jr z, wfk_wait_for_key          ;; infinite loop until a key is pressed
    pop hl                          ;; recovering the value of hl
    
    ld a, (hl)          ;; |
    ld d, a             ;; | e stores the keys pressed
    ld e, #10
    wfk_keydetection:
        xor #0xFF
        jr z, wfk_nopressedkeys
            ;; a key was pressed
            ld h, a         ;; |
            ld l, #0        ;; | Shift left 8
            ld d, #0
            dec e   ;; i-1
            add hl, de   ;; keypressed << 8 + (i-1)
            ld a, #01
            ret
        wfk_nopressedkeys:
        dec hl           ;; keys--
        ld a, (hl)       ;; restore the a content
        ld d, e          ;; d caches original pressed key
        dec e
    jr nz, wfk_keydetection
    xor a            ;; we set a to 00 to mark that no keys where pressed
    ret


;;=====================================================
;; It waits until we stop pressing the key
;; DESTROYS: AF, DE, BC
;;=====================================================
waitKeyReleased::
    wkr_wait_for_key_released:
        push hl
        call cpct_scanKeyboard_asm
        pop hl
        push hl
        call cpct_isKeyPressed_asm
        pop hl
    jr nz, wkr_wait_for_key_released 
    ret

;;===================================================================
;; This function is meant to be called inside an interruption handler (isr)
;; It plays music @ 50 Hz
;; The song needs to be initialized first
;;===================================================================
gs_nInterrupt: .db #6
FiftyHzMusicPlayer::
    ex af, af'      ;; | 
    push af         ;; | We need to save af
    ld hl, #gs_nInterrupt
    dec (hl) 
    jr nz, mn_isr_return
    
    ld (hl), #6 
    call cpct_akp_musicPlay_asm
    mn_isr_return:
    pop af        ;; 
    ex af, af'    ;; | Restoring af'
    ret 
    
;;=============================================
;; This function removes the interrupt handler and stops the music
;; DESTROYS: AF, AF’, BC, DE, HL, IX, IY
;;=============================================
StopMusicRmvIsr::
    call cpct_removeInterruptHandler_asm  ;; Removes Interrupt Handle (no more its needed).
    call cpct_akp_stop_asm      ;; Stops the music
    ld a, #6                    ;; |
    ld (gs_nInterrupt), a       ;; | reseting gs_nInterrupt to default
    ret

;;============================================================================
;;  It does nothing. Useful for functions that need a pointer to function
;;      but you do not want to do anything.
;;============================================================================
doNothing::
    ret

;;============================================================================
;;  When some function calls this function, what it does is return to 
;;      previous called function:
;;  function_a:
;;      call function_b
;;      ;; Do stuff in function_a
;;      ret
;;
;;  function_b:
;;      call function_c
;;      ;; Do stuff in function_b
;;      ret
;;
;;  function_c:
;;      call THIS_FUNCTION
;;      ret
;;
;;  Execution flow: 
;;      function_a calls function_b
;;      function_b calls function_c
;;      function_c calls THIS_FUNCTION
;;      THIS_FUNCTION returns to function_a
;;
;;  It is equivalent to do 'ret' in a function but doing it with call.
;;  Useful to generalize functions that do 'call' but some entity wants exit.
;;
;;  INPUTS:
;;       NOT IMPLEMENTED: 
;;          A: number of 'push' that have been executed in the function that
;;          is calling THIS_FUNCTION and 'pop' has not been executed.
;;  DESTROYS: AF, BC
;;============================================================================
returnToPreviousFunction::
    pop bc
    ;jr nz, returnToPreviousFunction
    ret

;;============================================================================
;;  INPUTS:
;;      A
;;  OUTPUTS:
;;      A: 0 if not greater than 0, 1 if greater than 0
;;============================================================================
greaterOrEqualThanZero::
    bit 7,a
    jr z, goetzok
        xor a
        ret
    goetzok:
        ld a, #1
        ret
    
;;============================================================================
;;  INPUTS:
;;      HL: array
;;       A: index
;;  OUTPUTS:
;;       A: element
;;============================================================================
getByteFromArray::
    add_hl_a
    ld a, (hl)
    ret