;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.include "gamemanager/serialization/serializeddata.h.s"
.include "entities/enemies/enemy.h.s"
.include "macros/cpct_undocumentedOpcodes.h.s"
.include "cpctglobals.h.s"
.include "globals.h.s"

.area _DATA
.area _CODE

;; There is an explicit definition of the types of enemies which is the following:
;; 00 : none
;; 01 : enemytype_1
;; 10 : enemytype_2
;; 11 : enemytype_3

;; There is also defined the max number of enemies per level, which can be up to 8
;; 00 10 01 11   00 00 00 00 
;; This leads us to a structure where each level stores 2 bytes of data with enemy information

;; At the same time, each enemy type defines their own locations and in every apparition on the level they make
;; aside extra properties defined in the enemy definition data
;;
;; For example, we could say that enemytype_1 is the mini saw:
;; Everytime we will read a 01 on the enemy distribution level data array, we'll know we are speaking about a mini saw
;; Once we resolved the type, we go to the specific enemy_array, and we'll read the location and some extra properties:
;;
;; mini_saw_appearances:
;; .dw #0x0505   	.db #00        .db #00
;;	^^^^^^^^^^   ^^^^^^^^^^^^^    ^^^^^^^^^^
;;   LOCATION 	 SAW DIRECTION   MOVEMENT UNITS
;;
;; If we have a enemy type that can have many skins, we can just simply add a skin to the appearance data for the specific entity
;; Each of this classes will have a manager on them, so spawning/de-spawning them will be up to the entity manager


;; ================================================================================================================
;; Returns in de the #level_enemy_data
;; Return: 
;; 		DE
;; ================================================================================================================
sd_get_level_enemy_data::
	ld de, #level_enemy_data
	ret


;; ================================================================================================================
;; Resets all the appearances indexes to restart the level enemy data
;; Destroys: HL 
;; ================================================================================================================
sd_reset_appearances_indexes:
	ld hl, #0000
	ld (char_jumper_appearances_index), hl
	ld (mini_saw_appearances_index), hl
	ld (xypawn_appearances_index), hl
	ret

;; ================================================================================================================
;; The char jumper data currently stands for 2 bytes of data so we do jumps 2 by 2.
;; Since the constructor only needs x and y to create an entity of this type
;; This function increases the appearance index and resolves it
;; DESTROYS: HL, IX, AF, DE
;; ================================================================================================================
sd_new_char_jumper::
	ld ix, #char_jumper_appearances
	ld hl, (char_jumper_appearances_index)
	ld a, h
	or a
	jr nz, sd_resolve_charjp_appearances
	ld a, l
	or a
	jr z, sd_spawn_jumper

	;; Based on the appearance index we sort out the positioning of this brand new spawned char jumper
	sd_resolve_charjp_appearances:
		inc ix
		inc ix
		;; loop new iterations
		dec hl
		ld a, h
		or a
		jr nz, sd_resolve_charjp_appearances
		ld a, l
		or a
	jr nz, sd_resolve_charjp_appearances

	sd_spawn_jumper:
	;; Dumping the content of ix on de
	ld d, 0(ix) 		;; spawn x axis
	ld e, 1(ix) 		;; spawn y axis

	call charjp_spawnJumper
	ld hl, (char_jumper_appearances_index)
	inc hl 
	ld (char_jumper_appearances_index), hl
	ret


;; ================================================================================================================
;; The char jumper data currently stands for 2 bytes of data so we do jumps 2 by 2.
;; Since the constructor only needs x and y to create an entity of this type
;; This function increases the appearance index and resolves it
;; DESTROYS: HL, IX, AF, DE
;; ================================================================================================================
sd_new_mini_saw::
	ld ix, #mini_saw_appearances
	ld hl, (mini_saw_appearances_index)
	ld a, h
	or a
	jr nz, sd_resolve_saw_appearances
	ld a, l
	or a
	jr z, sd_spawn_saw

	;; Based on the appearance index we sort out the positioning of this brand new spawned char jumper
	sd_resolve_saw_appearances:
		ld de, #5
		add ix, de
		;; loop new iterations
		dec hl
		ld a, h
		or a
		jr nz, sd_resolve_saw_appearances
		ld a, l
		or a
	jr nz, sd_resolve_saw_appearances

	sd_spawn_saw:

	call saw_spawnSaw		
	ld hl, (mini_saw_appearances_index)
	inc hl 
	ld (mini_saw_appearances_index), hl
	ret


;; ================================================================================================================
;; Resets all the appearances indexes to restart the level enemy data
;; Input: A num_map
;; Destroys: HL, AF, AF', DE
;; ================================================================================================================
sd_init_appareance_index_based_on_numap::  
	or a 			;; if 0 then we reset the appearances indexes
	call sd_reset_appearances_indexes
	ret z
	call sd_get_level_enemy_data 		;; de stores the level_enemy_data

	sd_indexes_loop:
		ex af, af'
		;; we need to go 2 bit by 2 bit sorting out the enemy type
		;; 00 -> none , 01 -> e_type1, 10 -> e_type2, 11 -> e_type3
		ld hl, #ENEMIES_PER_LEVEL  ;; We'll use a as the loop counter
		;; For convinience each 8 bit group gets iterated in a reverse way	
		sd_init_app_loop:
			push hl 
			dec l 		;; We normalise it to be in 0-7 range
			
			;; hl holds the position of the 2 bits we want to retrieve
			push de
			call cpct_get2Bits_asm
			;; at this point we have in l the value that corresponds with the ENEMIES_PER_LEVEL-l position
			ld a, l 								;; we transfer this value to a for convenience
			cp #ENEMIES_TYPE_CHARJUMPER 			;; if we are working with a char jumper, we simply look on the appareance table
			jr z, sd_char_jumper_appearance 		;; we jump to char jumper appearance	
			cp #ENEMIES_TYPE_SAW 					;; if we are working with a char jumper, we simply look on the appareance table
			jr z, sd_saw_appearance 				;; we jump to char jumper appearance
			cp #ENEMIES_TYPE_XYPAWN 				;; if we are working with a char jumper, we simply look on the appareance table
			jr z, sd_xypawn_appearance 				;; we jump to char jumper appearance
			jr sd_next_enemy 						;; if enemy type couldn't be resolved we assume it means no enemy
				
				sd_char_jumper_appearance:
				ld hl, (char_jumper_appearances_index) 		;; refreshing char jumper appearances ...
				inc hl
				ld (char_jumper_appearances_index), hl 
				jr sd_next_enemy

				sd_saw_appearance:
				ld hl, (mini_saw_appearances_index)
				inc hl
				ld (mini_saw_appearances_index), hl 
				jr sd_next_enemy

				sd_xypawn_appearance:
				ld hl, (xypawn_appearances_index)
				inc hl
				ld (xypawn_appearances_index), hl 

			sd_next_enemy:
			pop de
			pop hl
			dec l
		jr nz, sd_init_app_loop

		ex af, af'
		inc de  ;; |
		inc de  ;; | 2 bytes per level
		dec a
	jr nz, sd_indexes_loop
	ret


;;=================================================================================================================
;; 	level_enemy_data stands for the enemy type appearances per level, up to 8 enemies per level, each enemy is 2 bits
;;	00: nothing
;;	01: jumper
;;	10:	mini saw
;;	11: pawn xy
;;=================================================================================================================
level_enemy_data:
	.dw #0b0000000000000000 ;; level 0 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000101000000000 ;; level 1 data 00001010 00000000 => 0,0,2,2,0,0,0,0 => 0,0,0,0,2,2,0,0
	.dw #0b0100000000000000 ;; level 2 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b0000000000000000 ;; level 3 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b1010100000000000 ;; level 4 data 10101000 00000000 => 2,2,2,0,0,0,0,0 => 0,0,0,0,0,2,2,2
	.dw #0b0000000000000000 ;; level 5 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 6 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0100000000000000 ;; level 7 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b1001000000000000 ;; level 8 data 10010000 00000000 => 2,1,0,0,0,0,0,0 => 0,0,0,0,0,0,1,2
	.dw #0b0100000000000000 ;; level 9 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b0100000000000000 ;; level 10 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b0000000000000000 ;; level 11 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b1000000000000000 ;; level 12 data 10000000 00000000 => 2,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,2
	.dw #0b0100000000000000 ;; level 13 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b1010100000000000 ;; level 14 data 10101000 00000000 => 2,2,2,0,0,0,0,0 => 0,0,0,0,0,2,2,2
	.dw #0b0000000000000000 ;; level 15 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 16 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 17 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 18 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b1000000000000000 ;; level 19 data 10000000 00000000 => 2,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,2
	.dw #0b1010000000000000 ;; level 20 data 10100000 00000000 => 2,2,0,0,0,0,0,0 => 0,0,0,0,0,0,2,2
	.dw #0b1010000000000000 ;; level 21 data 10100000 00000000 => 2,2,0,0,0,0,0,0 => 0,0,0,0,0,0,2,2
	.dw #0b0000000000000000 ;; level 22 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 23 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b1001000000000000 ;; level 24 data 10010000 00000000 => 2,1,0,0,0,0,0,0 => 0,0,0,0,0,0,1,2
	.dw #0b0000000000000000 ;; level 25 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b1000000000000000 ;; level 26 data 10000000 00000000 => 2,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,2
	.dw #0b0100000000000000 ;; level 27 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b1000000000000000 ;; level 28 data 10000000 00000000 => 2,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,2
	.dw #0b1010010000000000 ;; level 29 data 10100100 00000000 => 2,2,1,0,0,0,0,0 => 0,0,0,0,0,1,2,2
	.dw #0b1010010000000000 ;; level 30 data 10100100 00000000 => 2,2,1,0,0,0,0,0 => 0,0,0,0,0,1,2,2
	.dw #0b0000000000000000 ;; level 31 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0100000000000000 ;; level 32 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b1000000000000000 ;; level 33 data 10000000 00000000 => 2,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,2
	.dw #0b0100000000000000 ;; level 34 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b0100000000000000 ;; level 35 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b1010000000000000 ;; level 36 data 10100000 00000000 => 2,2,0,0,0,0,0,0 => 0,0,0,0,0,0,2,2
	.dw #0b1000000000000000 ;; level 37 data 10000000 00000000 => 2,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,2
	.dw #0b0000000000000000 ;; level 38 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 39 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0100000000000000 ;; level 40 data 01000000 00000000 => 1,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,1
	.dw #0b0000000000000000 ;; level 41_0 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 41_1 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 41_2 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 41_3 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b1001000000000000 ;; level 41 data 10010000 00000000 => 2,1,0,0,0,0,0,0 => 0,0,0,0,0,0,1,2
	.dw #0b1010000000000000 ;; level 42 data 10100000 00000000 => 2,2,0,0,0,0,0,0 => 0,0,0,0,0,0,2,2
	.dw #0b1000000000000000 ;; level 43 data 10000000 00000000 => 2,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,2
	.dw #0b1010000000000000 ;; level 44 data 10100000 00000000 => 2,2,0,0,0,0,0,0 => 0,0,0,0,0,0,2,2
	.dw #0b0000000000000000 ;; level 45 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 46 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b1010000000000000 ;; level 47 data 10100000 00000000 => 2,2,0,0,0,0,0,0 => 0,0,0,0,0,0,2,2
	.dw #0b0000000000000000 ;; level 48 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0

	.dw #0b0000000000000000 ;; level 98 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 99 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	.dw #0b0000000000000000 ;; level 100 data 00000000 00000000 => 0,0,0,0,0,0,0,0 => 0,0,0,0,0,0,0,0
	;;...


;; ================================================================================================================
;; AI that jumps and follows the enemy
;; REMEMBER: MAX 2 per level.
;; ================================================================================================================
;; Data type is: ENEMIES_TYPE_CHARJUMPER
char_jumper_appearances_index: .dw #0000 
char_jumper_appearances:
	.db #038, #104 		;; x: 038 	y: 104	LVL 2 appearance
	.db #040, #120 		;; x: 040 	y: 120	LVL 7 appearance
	.db #042, #176		;; x: 042 	y: 176	LVL 8 appearance
	.db #022, #152		;; x: 022 	y: 152	LVL 9 appearance
	.db #072, #144		;; x: 072 	y: 144	LVL 10 appearance
	.db #040, #168		;; x: 040 	y: 168	LVL 13 appearance
	.db #040, #170		;; x: 040 	y: 170	LVL 24 appearance
	.db #040, #170		;; x: 040 	y: 170	LVL 27 appearance
	.db #016, #176		;; x: 016 	y: 176	LVL 29 appearance
	.db #032, #172		;; x: 032 	y: 172	LVL 30 appearance
	.db #068, #176		;; x: 068 	y: 176	LVL 32 appearance
	.db #020, #176		;; x: 020 	y: 176	LVL 34 appearance
	.db #040, #032		;; x: 040 	y: 032	LVL 35 appearance
	.db #040, #120		;; x: 040 	y: 120	LVL 40 appearance
	.db #040, #120		;; x: 040 	y: 120	LVL 41 appearance




	;; ...


;; ================================================================================================================
;; The minisaw exposed in the documentation above
;; ================================================================================================================
;; index of current appareance, we use a dw just in case the index does above what a byte can store 
;; Data type is: ENEMIES_TYPE_SAW
mini_saw_appearances_index: .dw #0000 
mini_saw_appearances:
	;; 	X		Y 		AXIS	RANGE 	SPEED
	.db #36, 	#95,	#1,	 	#0,		#0			;;|
	.db #12, 	#95,  	#1, 	#0,  	#0			;;| Level 1

	.db #48, 	#24,  	#2, 	#82,  	#1			;;| 
	.db #76,	#64,	#1,		#24,	#2			;;|
	.db #76,	#88,	#1,		#24,	#2			;;| Level 4

	.db #24,	#24,	#2,		#28,	#1			;;| Level 8

	.db #76,	#40,	#1,		#32,	#1			;;| Level 12

	.db #24,	#144,	#2,		#28,	#1			;;| 
	.db #40,	#144,	#2,		#32,	#1			;;| 
	.db #52,	#128,	#0,		#12,	#3			;;| Level 14

	.db #20,	#168,	#2,		#20,	#2			;;| Level 19

	.db #24,	#136,	#2,		#20,	#2			;;| 
	.db #52,	#136,	#2,		#20,	#2			;;| Level 20

	.db #0,		#152,	#3,		#72,	#1			;;| 
	.db #52,	#80,	#1,		#20,	#1			;;| Level 21

	.db #56,	#96,	#2,		#16,	#1			;;| Level 24

	.db #28,	#176,	#0,		#48,	#1			;;| Level 26

	.db #40,	#184,	#0,		#20,	#2			;;| Level 28

	.db #44,	#24,	#2,		#32,	#1			;;|
	.db #60,	#56,	#3,		#32,	#1			;;| Level 29

	.db #60,	#112,	#2,		#40,	#1			;;|
	.db #72,	#144,	#1,		#32,	#3			;;| Level 30

	.db #68,	#168,	#2,		#16,	#1			;;| Level 33

	.db #20,	#24,	#2,		#64,	#1			;;|
	.db #60,	#24,	#2,		#64,	#1			;;| Level 36
	
	.db #28,	#80,	#0,		#48,	#1			;;| Level 37
	
	.db #56,	#40,	#2,		#32,	#2			;;| Level 41

	.db #0,		#80,	#0,		#76,	#1			;;|
	.db #76,	#72,	#1,		#76,	#1			;;| Level 42

	.db #40,	#40,	#2,		#40,	#1			;;| Level 43

	.db #60,	#56,	#0,		#16,	#2			;;|
	.db #74,	#144,	#1,		#16,	#2			;;| Level 44

	.db #44,	#56,	#2,		#40,	#1			;;|
	.db #48,	#160,	#0,		#20,	#1			;;| Level 47


;; ================================================================================================================
;; Pawns that move right to left, up to down, like a caroussel, this appearance type requires a sprite definition
;; for variety
;; ================================================================================================================
;; Data type is: ENEMIES_TYPE_XYPAWN
xypawn_appearances_index: .dw #0000 
xypawn_appearances:


;; ================================================================================================================
;; Returns the 
;; Input: 
;;		A => num_map
;; Returns: DE -> x and y position of the character based on the level
;; DESTROYS: AF
;; ================================================================================================================
sd_getheroposition::
	ld ix, #hero_positions_level	;; hl contains the hero_position_leveñ array
	or a 							;; if it is 0 then we go to sd_hero_map_pos_solved
	jr z, sd_resolve_hero_pos
	sd_loop_hero_pos:
		inc ix 						;; | we go 4 by 4 bytes due to the fact we store 2 positions each x and y 
		inc ix 						;; |
		dec a
	jr nz, sd_loop_hero_pos
	sd_resolve_hero_pos:
	ld d, 0(ix)
	ld e, 1(ix)
	ret

;; ================================================================================================================
;; Spawn poisition for our hero in every level
;; ================================================================================================================
hero_positions_level:
	.db #5, #80 		;; lvl0 locations - left final
	.db #5, #50 		;; lvl1 locations - left final
	.db #5, #50 		;; lvl2 locations - left final
	.db #0, #130 		;; lvl3 locations - left final
	.db #0, #170 		;; lvl4 locations - left final
	.db #0, #170 		;; lvl5 locations - left final
	.db #0, #160 		;; lvl6 locations - left final
	.db #0, #125 		;; lvl7 locations - left final
	.db #0, #180 		;; lvl8 locations - left final
	.db #70, #140 		;; lvl9 locations - left final
	.db #00, #180 		;; lvl10 locations - left final
	.db #00, #30 		;; lvl11 locations - left final
	.db #0, #170 		;; lvl12 locations - left final
	.db #10, #180 		;; lvl13 locations - left final
	.db #0, #180 		;; lvl14 locations - left final
	.db #0, #180		;; lvl15 locations - left final
	.db #40, #35 		;; lvl16 locations - left final
	.db #40, #20 		;; lvl17 locations - left final
	.db #40, #35		;; lvl18 locations - left final
	.db #0, #180		;; lvl19 locations - left final
	.db #10, #180 		;; lvl20 locations - left final
	.db #75, #170 		;; lvl21 locations - left final
	.db #12, #184 		;; lvl22 locations - left final
	.db #00, #176 		;; lvl23 locations - left final
	.db #00, #176 		;; lvl24 locations - left final
	.db #00, #040 		;; lvl25 locations - left final
	.db #20, #176 		;; lvl26 locations - left final 
	.db #12, #20		;; lvl27 locations - left final 
	.db #0, #176 		;; lvl28 locations - left final 
	.db #0, #40 		;; lvl29 locations - left final 
	.db #0, #184 		;; lvl30 locations - left final
	.db #0, #144 		;; lvl31 locations - left final 
	.db #20, #160 		;; lvl32 locations - left final 
	.db #12, #184 		;; lvl33 locations - left final  
	.db #0, #184 		;; lvl34 locations - left final  
	.db #0, #40 		;; lvl35 locations - left final
	.db #0, #40 		;; lvl36 locations - left final
	.db #0, #96 		;; lvl37 locations - left final
	.db #36, #20 		;; lvl38 locations - left final
	.db #40, #24 		;; lvl39 locations - left final 
	.db #0, #124 		;; lvl40 locations - left final
	.db #0, #40 		;; lvl41_0 locations - left final
	.db #0, #180 		;; lvl41_1 locations - left final
	.db #0, #32 		;; lvl41_2 locations - left final
	.db #0, #180 		;; lvl41_2 locations - left final 
	.db #0, #124 		;; lvl41 locations - left final 
	.db #0, #48 		;; lvl42 locations - left final 
	.db #0, #120 		;; lvl43 locations - left final  
	.db #0, #144 		;; lvl44 locations - left final  
	.db #0, #180 		;; lvl45 locations - left final
	.db #0, #40 		;; lvl46 locations - left final
	.db #0, #172 		;; lvl47 locations - left final
	.db #0, #142 		;; lvl48 locations - left final
	
	.db #76, #168 		;; lvl98 locations - left final 
	.db #76, #176 		;; lvl99 locations - left final
	.db #76, #56 		;; lvl100 locations - left final - End of Game





