;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.area _DATA
.area _CODE

.include "cpctglobals.h.s"
.include "globals.h.s"

.include "mapmanager.h.s"
.include "entities/enemies/enemy.h.s"
.include "gamemanager/serialization/serializeddata.h.s"
.include "entities/hero.h.s"
.include "gameplaystatics.h.s"
.include "renderutils.h.s"
.include "keyboard/keyboard.h.s"
.include "music/InGameMusic.h.s"
.include "music/Fanfarria_2.h.s"
.include "entities/character.h.s"



finish_game: .db #NEXTLEVEL 	;; If == 1 end the game and return to the menu
gm_level_string: .asciz "000" 	;; we reuse it for the current level and for the current lives
gm_level_prefix: .asciz "LEVEL-"
gm_end_game_msg_one: .asciz " YOU WIN!"
gm_end_game_msg: .asciz "CONGRATULATIONS!"
gm_princess_unlocked: .asciz "QUEEN UNLOCKED"
gm_end_gameover_msg: .asciz "GAME OVER!"
gm_context_menu_resu: .asciz "1.Resume"
gm_context_menu_rest: .asciz "2.Restart"
gm_context_menu_quit: .asciz "3.Quit"


;;==========================================================================================================================
;; Sets finish game variable
;; INPUT: A
;;==========================================================================================================================
gm_setFinishGame_state::
	ld (finish_game), a
	ret

;;===================================================================
;; Function that will take care of playing the music and anims
;; of the menu thanks to the continuous interruptions
;; DESTROYS: AF, BC, DE, HL
;;===================================================================
gm_interruptHandler:
  ;; Plays the music
  jp FiftyHzMusicPlayer


;;===================================================================
;; This function returns 00 if the game wasn't completed and 01 if it was
;; DESTROYS: AF
;; RETURN: A => Completed?
;;===================================================================
game_was_completed_once: .db #00
gm_getGameWasCompleted::
	ld a, (game_was_completed_once)
	ret


;;===================================================================
;; Handles the end game, game over or last level!
;; DESTROYS: AF, BC, DE, HL, IY
;; INPUT A => 00 GAME OVER, 01 END GAME 
;;===================================================================
gm_Handle_EndGame:
	or a 							;; check if it consists on a game over or on a end game events
	jr z, gm_was_gameover			;; if it is a game over we jump to that bit of code
		;;
		;; We finished the game, yay
		;;
		call gm_last_lvl_cinematic
		ld    hl, #0x000F                  ;; H = Background PEN (0)
	  	ld    bc, #0x640A                   ;; text position y
	 	ld   iy, #gm_end_game_msg       	;; IY = Pointer to the start of the string
	 	ld a, (game_was_completed_once)
	 	or a
	 	jr nz, gm_handle_end_game_cont

	 		;; First time we completed the game
	 		call ru_drawText    
	 		ld d, #100
    		call waitNVSYNCs
    		call ru_fadeIn			;; Clearing the screen with a color
    		call ru_fadeOut
			ld    hl, #0x000F                  ;; H = Background PEN (0)
	  		ld    bc, #0x640A                   ;; text position y
	 		ld   iy, #gm_princess_unlocked       	;; IY = Pointer to the start of the string
	 		ld a, #01 							;; game was completed
	 		ld (game_was_completed_once), a 	;; |
	 		jr gm_handle_end_game_cont
	 	
 	
 	gm_was_gameover:
 	ld    hl, #0x000F                  ;; H = Background PEN (0)
  	ld    bc, #0x6414                   ;; text position y
 	ld   iy, #gm_end_gameover_msg       	;; IY = Pointer to the start of the string
 	
 	gm_handle_end_game_cont:
 	call ru_drawText    
    ld d, #500
    call waitNVSYNCs
    ld a, #MAINMENU
	jp gm_setFinishGame_state


;;==========================================================================================================================
;; This function gets called at the begining of every level, make sure to init all the required values to start a new level
;; DESTROYS: BC, DE, AF, HL
;;==========================================================================================================================
gm_initializegame:
    call StopMusicRmvIsr     		;; Stopping the music and removing the previous interrupt handler

	call cpct_waitVSYNC_asm
   	call ru_getCurrentVM
	ld bc, #0x4000 					;; |
	xor a 							;; | Color to drawn on all the vid memory
	call cpct_memset_asm 			;; Clearing the screen with a color

	ld hl, #0x1410
    call cpct_setPALColour_asm
	
	;; Destroy previous level leftover enemies
	call charjp_destroyAllJumpers 	;; Destroys all jumpers
	call saw_destroyAllSaws
	;; TODO: destroy all pawnsxy
	;; Level number interface feedback
	
	;; we check if we finished the game
	call mm_GetA_num_map 			;; We get on a the num map
	ld b, #MAX_LEVEL_NUM
	cp b
	ld a, #01 						;; 01 for the handler means we ended the game, yay!
	jr z, gm_Handle_EndGame
	call hero_get_num_lives
	cp #-1                          ;; we check if our current lives are -1, if it is -1 we cannot decrease it more hence we died :(
    ld a, #0                       	;; we pass 00 because the handler needs to know if it's game over or end game
	jr z, gm_Handle_EndGame

	;; if we didn't then we keep going
	call mm_GetA_num_map 			;; We get on a the num map
	inc a 							;; For the user level 0 is level 1 (numbers start at 1 for normal people hehe)
	ld h, #0 						;; we pass it to hl
	ld l, a
	ld de, #gm_level_string 		;; de holds the string reference to modify
  	call Num2Dec8b 					;; we convert the number in string
	ld    hl, #0x000F                  ;; H = Background PEN (0)
  	ld    bc, #0x642D                    ;; text position y
 	ld   iy, #gm_level_string       ;; IY = Pointer to the start of the string
  	call ru_drawText
	ld    hl, #0x000F                  ;; H = Background PEN (0)
  	ld    bc, #0x6414                    ;; text position y
 	ld   iy, #gm_level_prefix       ;; IY = Pointer to the start of the string
 	call ru_drawText
    ld d, #40
    call waitNVSYNCs

	ld a, #CURRENTLVLINPROGRESS 	;; |
	ld (finish_game), a 	 		;; | Setting finish game to 0 (game in progress)
	ret


;;==========================================================================================================================
;; This function deserializes the enemies based on the level we are on and spawns them as necesary
;; DESTROYS: BC, DE, AF, HL, IX
;;==========================================================================================================================
gm_initializeEnemies:
	call mm_GetA_num_map			;; We have in a the number of the map
	call sd_get_level_enemy_data   	;; We have in de the level_enemy_data first mem address
	or a 							;; In case the level is the first one, we resolve appareances directly
	jr z, resolve_appareances

		;; The level is not the 1st one
		resolve_enemy_data_level_address:
			inc de 			;; |
			inc de 			;; | 2 bytes per level 
			dec a
		jr nz, resolve_enemy_data_level_address
		;; At this point, we have sorted out the enemy data level address
	
	resolve_appareances:
	;; we need to go 2 bit by 2 bit sorting out the enemy type
	;; 00 -> none , 01 -> e_type1, 10 -> e_type2, 11 -> e_type3
	ld hl, #ENEMIES_PER_LEVEL  ;; We'll use a as the loop counter 7 to 0

	;; For convinience each 8 bit group gets iterated in a reverse way	
	gm_init_enemy_loop:
		push hl 
		dec l 		;; We normalise it to be in 0-7 range
		
		;; hl holds the position of the 2 bits we want to retrieve
		push de
		call cpct_get2Bits_asm
		;; at this point we have in l the value that corresponds with the ENEMIES_PER_LEVEL-l position
		ld a, l 								;; we transfer this value to a for convenience
		cp #ENEMIES_TYPE_CHARJUMPER 			;; if we are working with a char jumper, we simply look on the appareance table
		jr z, gm_init_char_jumper_appearance 	;; we jump to char jumper appearance	
		cp #ENEMIES_TYPE_SAW 					;; if we are working with a char jumper, we simply look on the appareance table
		jr z, gm_init_saw_appearance 			;; we jump to char jumper appearance
		cp #ENEMIES_TYPE_XYPAWN 				;; if we are working with a char jumper, we simply look on the appareance table
		jr z, gm_init_xypawn_appearance 		;; we jump to char jumper appearance
		jr gm_init_next_enemy 					;; if enemy type couldn't be resolved we assume it means no enemy
			
			gm_init_char_jumper_appearance:
			call sd_new_char_jumper
			jr gm_init_next_enemy

			gm_init_saw_appearance:
			call sd_new_mini_saw
			jr gm_init_next_enemy

			gm_init_xypawn_appearance:
			;; TODO, do the call to the xypawn_appearance


		gm_init_next_enemy:
		pop de
		pop hl
		dec l
	jr nz, gm_init_enemy_loop
	ret


;;==========================================================================================================================
;; Draws a mid-game contextual menu that lets the user restart, quit or resume the level
;; DESTROYS: BC, DE, AF, HL, IY
;;==========================================================================================================================
gm_drawContextMenu::
	call ru_drawContextMenuFrame
	ld    hl, #0x000F                   ;; H = Background PEN (0)
  	ld    bc, #0x5014                     ;; text position x
 	ld   iy, #gm_context_menu_resu   ;; IY = Pointer to the start of the string
 	call ru_drawText
 	ld    hl, #0x000F                   ;; H = Background PEN (0)
  	ld    bc, #0x5F14                     ;; text position x
 	ld   iy, #gm_context_menu_rest   ;; IY = Pointer to the start of the string
 	call ru_drawText
	ld    hl, #0x000F                   ;; H = Background PEN (0)
  	ld    bc, #0x6E14                     ;; text position x
 	ld   iy, #gm_context_menu_quit   ;; IY = Pointer to the start of the string
 	call ru_drawText

 	gm_context_menu_key_handling:
 		;; Check for Key '1' being pressed
 		call cpct_scanKeyboard_asm

	    ld      hl, #Key_1              ;; HL=Key_1 Keycode
	    call    cpct_isKeyPressed_asm   ;; Check if Key_1 is pressed
	    or      a                      ;; Check A == 0
	    jr      z, gm_one_not_pressed        ;; Resume game if 1 was pressed

        ;; 1 is pressed
        ;call cpct_waitVSYNC_asm
        ;call mm_drawFullCurrentMap
        ;call cpct_waitVSYNC_asm
        ;ret
		ld hl, #cpct_waitVSYNC_asm
		push hl
		ld hl, #mm_drawFullCurrentMap
		push hl
		jp cpct_waitVSYNC_asm

	    gm_one_not_pressed:

	    ;; Check for Key '2' being pressed
	    ld      hl, #Key_2              ;; HL=Key_2 Keycode
	    call    cpct_isKeyPressed_asm   ;; Check if Key_2 is pressed
	    or      a                      ;; Check A == 0
	    jr      z, gm_two_not_pressed        ;; Restart the level if 2 was pressed

	        ;; 2 is pressed
			ld a, #RESTART
			ld (finish_game), a   
			ret    
	   		
	   		gm_two_not_pressed:

	    ;; Check for Key '3' being pressed
	    ld      hl, #Key_3               ;; HL=Key_3 Keycode
	    call    cpct_isKeyPressed_asm   ;; Check if Key_3 is pressed
	    or      a                      ;; Check a == 0
	    jr      z, gm_context_menu_key_handling    ;; Go to main menu if 3 was pressed

	        ;; 3 is pressed
	        ld a, #MAINMENU
			ld (finish_game), a 
			ret

 	jr nz, gm_context_menu_key_handling

	ret


gm_intializehero:
	;call hero_reset_values
	;call mm_GetA_num_map
    ;call sd_getheroposition
    ;call hero_setPosition
    ;ret
	ld hl, #hero_setPosition
	push hl
	ld hl, #sd_getheroposition
	push hl
	ld hl, #mm_GetA_num_map
	push hl
	jp hero_reset_values


;;==========================================================================================================================
;; Main game loop
;; DESTROYS: BC, DE, AF, HL, IX
;;==========================================================================================================================
gm_game_loop:
	;; If we want to go to the menu we ret directly
	ld a, (finish_game)
	ld b, #MAINMENU
	cp b
	ret z

	;; Reseting indexes of appearances
	call mm_GetA_num_map
	call sd_init_appareance_index_based_on_numap
	;; Initializing enemies of type
	call gm_initializeEnemies
	;; Initializing hero values and position
	call gm_intializehero
	
	call mm_DecompressAndDrawFullCurrentMap		;; Decompressing and drawning current map

	;; Init in game music
	ld de, #_game_music
	call cpct_akp_musicInit_asm

	call cpct_waitVSYNC_asm
    halt
    halt
    call cpct_waitVSYNC_asm

    ld hl, #gm_interruptHandler
    call cpct_setInterruptHandler_asm

    ;;
    ;; Draw interface, current hp on both buffers
    ;;
  	ld h, #0 								;; we pass it to hl
	call hero_get_num_lives
	ld l, a
	ld de, #gm_level_string 						;; de holds the string reference to modify
  	call Num2Dec8b 									;; we convert the number in string
  	ld    hl, #0x000F                  				;; H = Background PEN (0)
  	ld    bc, #0x0423                    			;; text position x
 	ld   iy, #gm_level_string       				;; IY = Pointer to the start of the string
  	push bc
    call cpct_setDrawCharM0_asm   ;; Set draw char colours
    pop bc
    ld de, #0x8000
    call cpct_getScreenPtr_asm
    call cpct_drawStringM0_asm
	ld    hl, #0x000F                  				;; H = Background PEN (0)
  	ld    bc, #0x0423                    			;; text position x
 	ld   iy, #gm_level_string       				;; IY = Pointer to the start of the string
    push bc
    call cpct_setDrawCharM0_asm   ;; Set draw char colours
    pop bc
    ld de, #0xC000
    call cpct_getScreenPtr_asm
    call cpct_drawStringM0_asm


	;; We need to draw the interface in both buffers
  	ld   bc, #0x0423                            	;; C=Entity_X
  	ld de, #0xC000
  	call cpct_getScreenPtr_asm             ;; Get Pointer to Screen
    ex de, hl
  	;;Draw Sprite
  	call hero_getIdleSprite
  	ld bc, #0x0804
  	call cpct_drawSprite_asm 				;; here we draw the sprite relate to those lives
    
    ld   bc, #0x0423                            	;; C=Entity_X
    ld de, #0x8000
  	call cpct_getScreenPtr_asm             ;; Get Pointer to Screen
    ex de, hl
  	;;Draw Sprite
  	call hero_getIdleSprite
  	ld bc, #0x0804
  	call cpct_drawSprite_asm 				;; here we draw the sprite relate to those lives

    ;; ========================================================
	;; LEVEL LOOP, WHERE GAME LOGIC SHOULD BE CONTAINED
	;; ========================================================
	level_loop:
    	
		;;=====================================================
		;; Updating every entity
		;;=====================================================
		call hero_update            ;; Updates the hero
		
		ld hl, #saw_updateSaw		;;
		call saw_doForAll			;; Updates all saws

		call charjp_updateForAll	;; Update behaviour of all charjp.
		call hero_getPtrIY
			
		;;=====================================================
		;; Redrawing background for every entity
		;;=====================================================
			call mm_drawTiles
			
			ld hl, #saw_drawTilesAll
			call saw_doForAll

			ld hl, #mm_drawTiles
			call charjp_doForAll

		;;=====================================================
		;; Drawing every entity
		;;=====================================================			
			call hero_getPtrIY
			call ru_drawMaskedEntity

			ld hl, #saw_drawAll
			call saw_doForAll

			ld hl, #ru_drawMaskedEntity
			call charjp_doForAll

		;;=====================================================
		;; Updating coordinates for every entity
		;;=====================================================			
			call hero_getPtrIY
			call char_update_positions

			ld hl, #char_update_positions
			call saw_doForAll
			
			ld hl, #char_update_positions
			call charjp_doForAll

		call cpct_waitVSYNC_asm
		call ru_switchBuffer     ;; Wait for the raster to trigger vsync
		
		ld a, (finish_game) 		;; we check if we want to finish the game or not
		cp #CURRENTLVLINPROGRESS 	;; We loop the level until it's completed
	jr z, level_loop
	ret


;;==========================================================================================================================
;; We pressed start on the menu
;; Level logic loop
;;==========================================================================================================================
gm_startgame::
	game_main_exec:
		call gm_initializegame
		call gm_game_loop

		;; TBD: fadeout between levels?

		ld a, (finish_game) 	;; we check if we want to finish the game or not
		cp #MAINMENU 			;; We loop the game until we want to go to the main menu
	jr nz, game_main_exec
	ret


;;==========================================================================================================================
;; Plays the last level cinematic
;;==========================================================================================================================
gm_last_lvl_cinematic:
	ld de, #0x1148 			;; OK!
	call hero_setPosition
	call hero_reset_values
	call hero_getIdleSprite
	call hero_getPtrIY
    ld Ent_spr_l(iy), l         ;; |
    ld Ent_spr_h(iy), h         ;; Set sprite to the idle anim
    call hero_end_game_flip

	
	call fading_effect_cine 			;; OK!
		ld d, #70
		call waitNVSYNCs

    
	call fading_effect_cine 			;; OK!
		;; set princess 1 sprite
        ld bc, #0x4008
        call ru_gretScreenPointerOnDEBasedOnPosition
        ld hl, #_princess_glass_broken
        ld bc, #0x1008
        call cpct_drawSprite_asm
        ld d, #70
		call waitNVSYNCs

	call fading_effect_cine 			;; OK!
		;; set princess 2 sprite
		ld bc, #0x4008
        call ru_gretScreenPointerOnDEBasedOnPosition
        ld hl, #_princess_free
        ld bc, #0x1008
        call cpct_drawSprite_asm
        ld bc, #0x420A
        call ru_gretScreenPointerOnDEBasedOnPosition
        ld hl, #_princess_sprite_0
        ld bc, #0x0804
        call cpct_drawSprite_asm

		ld de, #0x2C60
		call hero_setPosition
	    
		ld d, #70
		call waitNVSYNCs


	call fading_effect_cine
		ld bc, #0x4008
        call ru_gretScreenPointerOnDEBasedOnPosition
        ld hl, #_princess_free
        ld bc, #0x1008
        call cpct_drawSprite_asm
        ld bc, #0x6024
        call ru_gretScreenPointerOnDEBasedOnPosition
        ld hl, #_princess_sprite_0
        ld bc, #0x0804
        call cpct_drawSprite_asm
		ld d, #200
		call waitNVSYNCs

		;; Last fanfarria
		ld de, #_end_music
		call cpct_akp_musicInit_asm
		call cpct_waitVSYNC_asm
	    halt
	    halt
	    call cpct_waitVSYNC_asm
	    ld hl, #gm_interruptHandler
	    call cpct_setInterruptHandler_asm
		call ru_drawContextMenuFrame
	 	ld    hl, #0x000F                   ;; H = Background PEN (0)
	  	ld    bc, #0x5F14                     ;; text position x
	 	ld   iy, #gm_end_game_msg_one   ;; IY = Pointer to the start of the string
	 	call ru_drawText
		ld d, #500
		call waitNVSYNCs
	
	;call ru_fadeIn
	;call ru_fadeOut
	;ret
	ld hl, #ru_fadeOut
	push hl
	jp ru_fadeIn


fading_effect_cine:
	;call ru_fadeOut
    ;call cpct_waitVSYNC_asm
    ;call mm_drawFullCurrentMap
	;call hero_getPtrIY
	;call ru_drawMaskedEntity
	;call ru_switchBuffer
	;ret

	ld hl, #ru_switchBuffer
	push hl
	ld hl, #ru_drawMaskedEntity
	push hl
	ld hl, #hero_getPtrIY
	push hl
	ld hl, #mm_drawFullCurrentMap
	push hl
	ld hl, #cpct_waitVSYNC_asm
	push hl
	jp ru_fadeOut