;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.area _DATA
.area _CODE

.include "entities/entity.h.s"
.include "entities/hero.h.s"
.include "entities/character.h.s"
.include "cpctglobals.h.s"
.include "renderutils.h.s"
.include "gamemanager/mapmanager.h.s"
.include "gameplaystatics.h.s"
.include "globals.h.s"
.include "collisions/collisions.h.s"
.include "collisions/collisionsprofile.h.s"
.include "gamemanager/game.h.s"
.include "keyboard/keyboard.h.s"


;;===========================================
;;===========================================
;; PRIVATE DATA
;;===========================================
;;===========================================

;;
;; Hero animations
;;
idle_sprite: .dw _hero_sprites_0

;;
;; Hero data
;;
defineCharacter hero, 39, 100, 4, 8, ENT_ACTIVE, _hero_sprites_0, CHAR_GRAV_NORMAL

;;======================================
;; Returns in hl hl the idle sprite
;;======================================
hero_getIdleSprite::
    ld hl, (idle_sprite)
    ret

walking_anim:
    .dw     _hero_sprites_1     ;; walk1  0-1
    .dw     _hero_sprites_2     ;; walk2  2-3

inputDisabled: .db #00
hero_number_lives:  .db #HERO_MAX_LIVES

;; Custom keybinds for the hero move set
user_keyboard_definitions::
    .dw #Key_D,         #moveHeroRight
    .dw #Key_A,         #moveHeroLeft
    .dw #Key_Space,     #startJump 
    .dw #Key_Esc,       #gm_drawContextMenu
    

;;=============================================
;;=============================================
;; PUBLIC FUNCTIONS
;;=============================================
;;=============================================
hero_reset_values::
    ld a, (hero_look_side)
    cp #ANIM_Normal_right
    jr z, hero_look_side_is_correct
        call hero_flip_sprites_h
        ld a, #ANIM_Normal_right
        ld (hero_look_side), a  
    hero_look_side_is_correct:
    ld a, (hero_gravdirection)
    cp #CHAR_GRAV_FLIPPED
    jr nz, hero_grav_was_normal
        call hero_flip_sprites_v
        ld a, #CHAR_GRAV_NORMAL
        ld (hero_gravdirection), a
    hero_grav_was_normal:
    ld a, #ENT_ACTIVE
    ld (hero_isactive), a
    xor a
    ld (hero_frame), a 
    ld a, #-1
    ld (hero_jump_index), a 
    ld a, #CHAR_FALLING
    ld (hero_state), a 
    ld (hero_pstate), a
    ret 

;;=============================================
;; Sets the sprite based on A
;; INPUT:
;;      A: 00 sprite hero | 01 sprite princess
;; DESTROYS: AF, HL, BC, DE
;;=============================================
hero_set_sprite_mem::
    or a
    jr z, set_hero_sprite
        ;; set princess sprite
        ld hl, #_princess_sprite_0
        ld (hero_sprite), hl
        ld (idle_sprite), hl
        ld hl, #_princess_sprite_1
        ld ix, #walking_anim 
        ld 0(ix), l
        ld 1(ix), h
        ld hl, #_princess_sprite_2
        ld 2(ix), l
        ld 3(ix), h
        ;; Jumping functionality
        ld hl, #user_keyboard_definitions
        ld de, #10
        add hl, de
        ld de, #changeGravityDirection
        ld (hl), e
        inc hl
        ld (hl), d
        ret
    set_hero_sprite:
    ld hl, #_hero_sprites_0
    ld (hero_sprite), hl
    ld (idle_sprite), hl
    ld ix, #walking_anim 
    ld hl, #_hero_sprites_1
    ld 0(ix), l
    ld 1(ix), h
    ld hl, #_hero_sprites_2
    ld 2(ix), l
    ld 3(ix), h
    ;; Jumping functionality
    ld hl, #user_keyboard_definitions
    ld de, #10
    add hl, de
    ld de, #startJump
    ld (hl), e
    inc hl
    ld (hl), d
ret

;;=============================================
;; Returns in A the number of lives
;; RETURNS:
;;      A: number of lives
;;=============================================
hero_get_num_lives::
    ld a, (hero_number_lives)
    ret 


;;=============================================
;; Sets the number of lives
;; INPUT:
;;      A: number of lives
;;=============================================
hero_set_num_lives::
    ld (hero_number_lives), a
    ret 


;;=============================================
;; Gets a pointer to hero data in HL
;; This function will be the one called in the entity iterators to check collision w/ the character
;; DESTROYS: HL
;; RETURNS:
;;      HL: Pointer to Hero Data
;;=============================================
hero_getPtrHL::
    ld hl, #hero_data      ;; HL points to the hero data
    ret

;;=============================================
;; Gets a pointer to hero data in IY
;; DESTROYS: IY
;; RETURNS:
;;      IY: Pointer to Hero Data
;;=============================================
hero_getPtrIY::
    ld iy, #hero_data      ;; HL points to the hero data
    ret

;;=============================================
;;  Returns hero's XY position
;;  RETURNS:
;;       H: X position
;;       L: Y position
;;  DESTROYS: HL
;;=============================================
hero_getXYPosition::
    ld a, (hero_x)
    ld h, a
    ld a, (hero_y)
    ld l, a
    ret

anim_pause    : .db #00        ;; Frames we are going to pause between sprite changes
hero_anim_handler:
    
    ld a, (hero_x)              ;; | If hero didn't move on the x axis
    ld b, a                     ;; | Then set an idle animation
    ld a, (hero_px)             ;; |
    cp b                        ;; |
    jr z, hero_idle             ;; | If the hero state is walking then go to walking

        ;; HERO !IDLE, (not idling)
        ld a, (anim_pause)                  ;; A = current anim pause               
        inc a                               ;; A++                                  
        ld (anim_pause), a                  ;; current anim pause = A
        cp #08                              ;; if(current anim pause != 08)
        jr nz, end_anim                     ;; we don't call anim code

            ;; ANIMATION
            ld iy, #hero_data                  ;; ix has the hero data                
            ld hl, #walking_anim               ;; iy has the anim table
            ld a, #4                           ;; a is the frames of the animation 2*2 as specified on the function body
            call setLoopableAnimation          ;; We swap frames
            xor a                              ;; |
            ld (anim_pause), a                 ;; | rolling back anim pause to 0
            jp end_anim
         

    hero_idle:
    ld a, (hero_pstate)         ;; 
    ld b, a                     ;; | Loading in b the hero previous state
    ld a, (hero_state)          ;; Loading in a the hero current state
    cp b                        ;;
    jr z, end_anim              ;; If the state didn't change (since it is idle), then we don't do anything
    xor a  
    ld (hero_frame), a          ;; Resetting frames since we changed anim
    ld hl, (idle_sprite)     ;; |
    ld iy, #hero_data           ;; |
    ld Ent_spr_l(iy), l         ;; |
    ld Ent_spr_h(iy), h         ;; Set sprite to the idle anim

    end_anim:
        ld a, (hero_state)      ;; |
        ld (hero_pstate), a     ;; | Refreshing prev state
    ret

;;=============================================
;; Updates the Hero
;; DESTROYS: AF, BC, HL
;;=============================================
hero_update::
    call hero_gravity       ;; constant fall-down
    call jumpControl        ;; Do jumps
    call checkUserInput     ;; Check if user presses keys
    call wallJumpControl    ;; Handle wall jumping
    jr hero_anim_handler  ;; Animations and sprite selection


;;=============================================
;; Sets the hero position
;; INPUT: DE -> Hero position
;; DESTROYS: AF 
;;=============================================
hero_setPosition::
    ld a, d
    ld (hero_x), a
    ld a, e
    ld (hero_y), a
ret

;;===============================================================
;; Function to call when hero dies
;; DESTROYS: AF
;;===============================================================
hero_ondeath::
    ld a, (hero_number_lives)
    dec a                       ;; we decrease the num of lives we have missing
    ld (hero_number_lives), a   ;; |
    ld a, #RESTART              ;; and we restart the level
    jp gm_setFinishGame_state

hero_kill::
    jr hero_ondeath

;;=============================================
;;=============================================
;; PRIVATE FUNCTIONS
;;=============================================
;;=============================================

;;===============================================================
;; This function kicks in if the tile is not a blocking only tile
;;===============================================================
hero_response_to_tiles::
    ld a, l
    cp #RANGE_NEXT_MAP_SAME_LVL
    jr z, hero_next_map
    ld a, h                  
    cp #RANGE_NEXT_MAP_SAME_LVL      
    jr z, hero_next_map
    ld a, l
    cp #RANGE_DEATHLY
    jr z, hero_die
    ld a, h                  
    cp #RANGE_DEATHLY      
    jr z, hero_die
    ld a, l
    cp #RANGE_GRAVITY
    jr z, hero_gravity_response
    ld a, h                  
    cp #RANGE_GRAVITY      
    jr z, hero_gravity_response
    ret nz
        
        ;; Traveling the next map
        hero_next_map:
        call mm_NextMap
        ld a, #NEXTLEVEL
        call gm_setFinishGame_state
        ld a, #0xFF         ;; This is the interruption signal
        ret

        ;; Dying
        hero_die:
        call hero_ondeath
        ld a, #0xFF
        ret

        ;; Hero swap gravity
        hero_gravity_response:
        jp changeGravityDirection 


hero_gravity:
    ld a, (waitTicksGrav)
    or a
    jr z, skip_dec
        dec a
        ld (waitTicksGrav), a
    skip_dec:

    ld hl, #hero_response_to_tiles
    ld de, #hero_ondeath
    ld iy, #hero_data
    jp char_gravity

;;=============================================
;; Controls Jump movements
;; DESTROYS: AF, BC, HL, DE
;;=============================================
jumpControl:
    ld hl, #hero_response_to_tiles
    ld de, #hero_kill
    jp char_jumpControl


;;=============================================
;; This function flips all the hero sprites vertically
;; DESTROYS: HL, DE, IX, BC, AF
;;=============================================
hero_flip_sprites_v:
    ld hl, (idle_sprite)
    ld d, h 
    ld e, l
    ld bc, #28
    add hl, bc
    ld bc, #0x0804
    ex de, hl
    call cpct_vflipSprite_asm
    ld ix, #walking_anim
    ld e, 0(ix) 
    ld d, 1(ix)
    ld h, d
    ld l, e
    ld bc, #28
    add hl, bc
    ld bc, #0x0804
    ex de, hl
    call cpct_vflipSprite_asm
    ld e, 2(ix) 
    ld d, 3(ix)
    ld h, d
    ld l, e
    ld bc, #28
    add hl, bc
    ld bc, #0x0804
    ex de, hl
    jp cpct_vflipSprite_asm


;;=============================================
;; This function flips all the hero sprites horizontally
;; DESTROYS: HL, DE, IX, BC, AF
;;=============================================
hero_flip_sprites_h:
    ld bc, #0x0804
    ld hl, (idle_sprite)
    call cpct_hflipSpriteM0_asm
    ld ix, #walking_anim
    ld l, 0(ix) 
    ld h, 1(ix)
    ld bc, #0x0804
    call cpct_hflipSpriteM0_asm
    ld l, 2(ix) 
    ld h, 3(ix)
    ld bc, #0x0804
    jp cpct_hflipSpriteM0_asm


;;==========================================================================================
;; This function is used only on the last cinematic to flip the hero so it looks to the princess
;; DESTROYS: HL, DE, IX, IY, BC, AF
;;==========================================================================================
hero_end_game_flip::
    call hero_flip_sprites_h
    ld iy, #hero_data
    ;; we store in b the look side
    ld Char_look_side(iy), #ANIM_Normal_left
    ret

;;=============================================
;; Starts Hero Jump
;; DESTROYS: AF
;;=============================================
waitTicks      : .db #00
launchDirection: .db #00
wallJumping    : .db #00

startJump:
    ld iy, #hero_data
    ld hl, #checkWallJump
    jp char_startJump


checkWallJump::
    ld a, Ent_x(iy)
    ld b, a
    ld a, Ent_px(iy)
    cp b
    ret nz                   ;;if the hero moved this frame, no need to check for walljump

    ld iy, #hero_data
    ld hl, #entityIsCollidingWithTileRight
    call char_ProcessRightLeftGroundCollision
    ld a, #0
    ld (launchDirection), a
    jr z, doWallJump

    ld iy, #hero_data
    ld hl, #entityIsCollidingWithTileLeft
    call char_ProcessRightLeftGroundCollision
    ld a, #1
    ld (launchDirection), a
    ret nz

    doWallJump:
        sub a                   ;; |
        ld  (hero_jump_index), a    ;; | hero_jump_index = 0 (Activate jumping)
        ld a, #CHAR_JUMPING         ;; Refreshing hero state so now character is jumping up
        ld (hero_state), a          ;; |
        ld a, #1
        ld (wallJumping), a
        ld (inputDisabled), a
        
ret

wallJumpControl:
    ld a, (wallJumping)
    or a
    ret z

    ld a, (waitTicks)
    cp a, #WJ_TICKS
    jr nz, updateWallJump

    xor a
    ld (wallJumping), a
    ld (inputDisabled), a
    ld (waitTicks), a

    ret

    updateWallJump:

        inc a
        ld (waitTicks), a
        ld a, (launchDirection)
        or a
        jr z, moveHeroLeft
        
        jr moveHeroRight


;;=============================================
;; Move hero right if he is not at the limit
;; DESTROYS: AF, ED
;;=============================================
moveHeroRight:
    ld iy, #hero_data
    ;; we store in b the look side
    ld b, Char_look_side(iy)
    push bc
    ld hl, #hero_response_to_tiles
    ld bc, #doNothing
    call char_moveRight
    ;; we store in a the new look side
    ld a, Char_look_side(iy)
    pop bc
    cp b
    ret z
    jp hero_flip_sprites_h


;;=============================================
;; Move hero left if he is not at the limit
;; DESTROYS: AF, HL
;;=============================================
moveHeroLeft:
    ld iy, #hero_data
    ;; we store in b the look side
    ld b, Char_look_side(iy)
    push bc
    ld hl, #hero_response_to_tiles
    ld bc, #doNothing
    call char_moveLeft
    ;; we store in a the new look side
    ld a, Char_look_side(iy)
    pop bc
    cp b
    ret z
    jp hero_flip_sprites_h

;;==========================================================
;;	Function that changes the gravity direction
;;	DESTROYS: A
;;==========================================================
waitTicksGrav: .db #00
changeGravityDirection:
    ld a, (hero_state)           ;; If we are falling we don't let the user change the gravity direction
    cp #CHAR_FALLING
    ret z
    
    ld a, (waitTicksGrav)
    or a
    ret nz

    ld a, #5
    ld (waitTicksGrav), a
    ld a, (hero_gravdirection)      ;; Stores in a the gravity direction
    xor #01                         ;; |
    ld (hero_gravdirection), a      ;; | Swaps it based on the current content
    ld a, #CHAR_FALLING             ;; |
    ld (hero_state), a              ;; | We put the hero on a falling state
    jp hero_flip_sprites_v


;;=============================================
;; Returns on ix the user keyboard definition
;; RETURNS:
;;      IX => user keyboard definitions
;;=============================================
get_keyboard_ix::
    ld ix, #user_keyboard_definitions
    ret

;;=============================================
;; Checks user input input and reacts
;; DESTROYS:
;;=============================================
checkUserInput:
    ld a, (inputDisabled)               ;; if the input is disabled we just ret
    or a
    ret nz
    ;; Input isn't disabled, we process the keys
    call cpct_scanKeyboard_asm          ;; Scan the whole keyboard 
    ld ix, #user_keyboard_definitions   ;; ix will store the user keyboard definition array
    ld e, #4                            ;; e represents the number of input actions we have available
    cui_check_keyboard:
        ld l, 0(ix)                     ;; the first 2 bytes represent the pressed key
        ld h, 1(ix)                     ;; |
        call    cpct_isKeyPressed_asm 
        or a                            ;; if the key wasn't pressed we just check the next one
        jr z, cui_key_not_pressed 
            ;; a key was pressed
            ld l, 2(ix)                 ;; if a key was pressed the next two bytes represent the function to call
            ld h, 3(ix)                 ;; |
            ld (cui_func), hl           ;; |
            push ix                     ;; we need to store ix
            push de                     ;; we need to store de
            cui_func = . + 1            ;; cui_col will modify the line we have down there from right to left
            call doNothing
            pop de 
            pop ix
        cui_key_not_pressed:
        ld bc, #04                      ;; |
        add ix, bc                      ;; | the next key is 4 bytes ahead
        dec e       
    jr nz, cui_check_keyboard
    ret


    