;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.area _DATA
.area _CODE

.include "entities/enemies/enemy.h.s"
.include "../../gamemanager/mapmanager.h.s"
.include "macros/cpct_undocumentedOpcodes.h.s"
.include "renderutils.h.s"
.include "entities/hero.h.s"
.include "gameplaystatics.h.s"


;;===========================================================================================================================================================
;; REQUIREMENT:
;; Number of enemies of this entity should be something set on stone and fixed, since we are going to play directly with the mem addresses we have available
;; If the mem addresses reserved for this enemy get stepped by something else the program will work erraticly and will fail. 
;;===========================================================================================================================================================


;; In this case we can have maximum OF entities of this enemy in the same map, so we will have to keep TBD bytes
;; SAW SIZE 						= 		15 bytes
;; EXTRA BYTES FOR THIS ENTITY 		= 		0n bytes
saw_num_ent: 		.db 00
defineNMiniSawDefault saw_vector

saw_anim:
    .dw _minisaw_sprites_0
	.dw _minisaw_sprites_1

saw_tick:
	.db		#0

;;=============================================================
;; Executes (HL) code for all the saws
;; Inputs: 
;; 		HL => Pointer to the routine to execute
;;=============================================================
saw_doForAll::
	ld a, (saw_num_ent) 		
	or a 									;; | if there are no entities spawned we simply return
	ret z 									;; |
	ld iy, #SAW_ADDRESS 
	ld (saw_rountine_to_call), hl
	
	;; Warning: if weird behaviours, please track the content of iy, it might be modified on a subroutine
	;; called by the self modified code
	ld b, a
	saw_doThings_loop:
		push bc 							;; save af on the stack
		push iy
		saw_rountine_to_call = . + 1 	    ;; Auto modificable code, hl will store the function we'll call
		call ru_drawMaskedEntity			;; draws iy entity 
		pop iy
		ld bc, #SAW_SIZE 			        ;; number of bytes to jump
		add iy, bc 							;; iy stores the new mem address after the jump
		pop bc 								;; restore af		
		;dec a
	djnz saw_doThings_loop
	;jr nz, saw_doThings_loop

ret

saw_drawAll::
	ld a, Ent_draw(iy)
	or a
	ret z

	ld hl, #ru_drawMaskedEntity
	jr saw_doForAll

saw_drawTilesAll::
	ld a, Ent_draw(iy)
	or a
	ret z

	ld hl, #mm_drawTiles
	jr saw_doForAll

saw_anim_handler::
	ld a, Saw_anim(iy)
	or a
	jr z, switchSprite
		dec a
		ld Saw_anim(iy), a
		ret
	switchSprite:
		ld a, #4
		ld hl, #saw_anim

		call setLoopableAnimation
		
		ld Saw_anim(iy), #4

	ret

;;=============================================================
;; Adds one saw to the end of the array of minisaws
;; Inputs:
;;		IY => Pointer to Saw
;;=============================================================
saw_updateSaw::

	call saw_anim_handler
	
	ld a, Saw_speed(iy)
	or a
	jp z, check_colision_hero					;; if speed 0, we don't update coords

	ld b, Saw_tick(iy)
	cp b

	jp nz, skipUpdate

		ld b, Saw_spawn(iy)
		
		ld a, Saw_axis(iy)
		or a
		jr z, update_x_right 
		dec a
		jr z, update_x_left
		dec a
		jr z, update_y_up
		dec a
		jr z, update_y_down

		update_x_right:
			ld a, Saw_range(iy)
			add b

			ld b, a
			ld a, Ent_x(iy)
			cp b

			jr z, reset_x_r
			
				ld a, Ent_x(iy)
				inc a
				ld Ent_x(iy), a

				jr endUpdate 

			reset_x_r:
				ld a, Saw_spawn(iy)
				ld Ent_x(iy), a

				jr endUpdate

		update_x_left:
			ld a, Ent_x(iy)
			ld b, a
			ld a, Saw_spawn(iy)
			sub b

			ld b, a
			ld a, Saw_range(iy)
			cp b

			jr z, reset_x_l
			
				ld a, Ent_x(iy)
				dec a
				ld Ent_x(iy), a

				jr endUpdate 

			reset_x_l:
				ld a, Saw_spawn(iy)
				ld Ent_x(iy), a

				jr endUpdate

		update_y_up:
			ld a, Saw_range(iy)
			add b

			ld b, a
			ld a, Ent_y(iy)
			cp b

			jr z, reset_y_u

				ld a, Ent_y(iy)
				inc a
				ld Ent_y(iy), a
				
				jr endUpdate 
				
			reset_y_u:
				ld a, Saw_spawn(iy)
				ld Ent_y(iy), a

				jr endUpdate

		update_y_down:
			ld a, Ent_y(iy)
			ld b, a
			ld a, Saw_spawn(iy)
			sub b

			ld b, a
			ld a, Saw_range(iy)
			cp b

			jr z, reset_y_d

				ld a, Ent_y(iy)
				dec a
				ld Ent_y(iy), a
				
				jr endUpdate 
				
			reset_y_d:
				ld a, Saw_spawn(iy)
				ld Ent_y(iy), a
		
		endUpdate:
			xor a
			ld Saw_tick(iy), a
			ld a, #2
            ld Ent_draw(iy), a

			jr check_colision_hero

	skipUpdate:
		ld a, Saw_tick(iy)
		inc a
		ld Saw_tick(iy), a

	check_colision_hero:
		call hero_getPtrHL						;; HL = hero's data
		call isColliding						;; It checks if the hero and the charjp are colliding

		or a									;; Same that cp #0
		jp nz, hero_kill						;; If A != 0 it means that charjp is touching the hero, so charjp have to kill the hero

ret

;;=============================================================
;; Adds one saw to the end of the array of minisaws
;; Inputs: (Spawn parameters)
;; 		IX => Pointer to saw params
;;=============================================================
saw_spawnSaw::
	ld a, (saw_num_ent) 				;; A = number os saw entities spawned
	cp #SAW_MAX_ENT 					;; if saw_num_ent == SAW_MAX_ENT
	ret z 								;; we don't spawn one entity

	inc a 								;; If we are able to spawn an entity then
	ld (saw_num_ent), a 				;; we update the current number of entities we have spawned

	dec a 				 				;; we want current number - 1 since its an array getter

	call saw_getSawById					;; Based on the num_ent, we resolve the mem address we have to write at
	
	ld a, 0(ix)
	ld Ent_x(iy),  		a 					;; Setting defaults
	ld Ent_px(iy), 		a 					;; |
	ld Ent_ppx(iy),		a
	ld a, 1(ix)
	ld Ent_y(iy),  		a 					;; |
	ld Ent_py(iy), 		a 					;; |
	ld Ent_ppy(iy),		a
	ld a, 2(ix)
	ld Saw_axis(iy), 	a
	ld a, 3(ix)
	ld Saw_range(iy), 	a
	ld a, 4(ix)
	ld Saw_speed(iy),	a
	xor a
	ld Saw_tick(iy),	a
	
	ld a, 2(ix)
	or a
	jr z, x_axis
		cp #1
		jr z, x_axis
		
		ld a, 1(ix)
		ld Saw_spawn(iy), a
		ret
	x_axis:
	ld a, 0(ix)
	ld Saw_spawn(iy), a
ret


;;================================================================================
;; Gets the mem address of a saw saw giving an id
;; This function does not check boundaries
;; Inputs: (Spawn parameters)
;; 		A  => id of saw to get (from 0 to n-1)
;; Return:
;;		IY => Returns the memory address to the first byte of the requested saw
;; Destroys:
;; 		AF, BC, IY
;;================================================================================
saw_getSawById::
	ld iy, #SAW_ADDRESS 		        ;; HL holds the first mem address
	or a 								;; | If the id is 0 we can return directly on hl the first mem address
	ret z

	ld bc, #SAW_SIZE 			        ;; a stores the size we want to jump between mem addresses

	saw_resolve_mem_address: 		
		add iy, bc 						;; Keeping the result on hl will make sure that in the last iteration of this loop hl will hold the correct reference if the constants are set correctly
		dec a                           ;; |
	jr nz, saw_resolve_mem_address
ret



;;=============================================================
;; Destroys the minisaw given a mem address
;; Inputs:
;; 		DE => Mem address of saw to destroy
;;=============================================================
saw_destroySaw::
	ld a, (saw_num_ent) 				;; A = number os saw entities spawned
	or a 								;; if saw_num_ent == 0
	ret z 								;; if there are no entities we cannot destroy the entity
	
	;;
	;; Specifics of this function: 
	;; - Erases always the last element on the imaginary array of saw data
	;; - In order to erase a concrete element, it is needed first to do the following operations:
	;;		1. Copy the last element of the array in the mem address we want to erase
	;; 			aa, bb, cc, dd => aa, dd, cc, dd
	;;		2. Erase the last element of the array
	;; 			aa, dd, cc, dd => aa, dd, cc, xx
	;; - So it is needed to know precisely the size of the content to be able to copy it around mem addresses without corrupting anything
	;;

	;;========================================================================
	;; Copy the last element of the array in the mem address we want to erase
	;;========================================================================
	ld a, (saw_num_ent) 		;; a == Current number of entities, the previous check ensures it's over 0
	dec a 						;; we want current number - 1 since its an array getter
	call saw_getSawById  	;; By calling this function we have on iy the memory address we want to copy
	ld__d_iyh 					;; store in d, iyh which is the high part of the mem address we want to copy
	ld__e_iyl 				 	;; store in e, iyl which is the low part of the mem address we want to copy
	ex de, hl  					;; We store for ldir the content of iy on hl
	ld bc, #SAW_SIZE  	        ;; bc stores the 'class' size to transfer it later on with ldir
	ldir 						;; we copy hl on de, this means that the step one is completed, see previous comment

	;;========================================================================
	;; Erase the last element of the array
	;;========================================================================
	ld a, (saw_num_ent) 		;; First, we decrease the number of elements we'll track
	dec a 						;; |
	ld (saw_num_ent), a 		;; |

	;; We could zero out the last element of the array, but that will take out some cycles and bytes we need for other operations
	;; So by design, we are going to leave "dead data" in the last position of that memory address, the dead data will be 
	;; replaced by fresh data every time we will spawn new elements

ret


;;=============================================================
;; 'Destroys' all the minisaws
;; 	DESTROY: AF
;;=============================================================
saw_destroyAllSaws::
	xor a
	ld (saw_num_ent), a
	ret