;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.area _DATA
.area _CODE

.include "entities/enemies/enemy.h.s"
.include "macros/cpct_undocumentedOpcodes.h.s"
.include "renderutils.h.s"
.include "gameplaystatics.h.s"
.include "entities/character.h.s"
.include "gamemanager/game.h.s"
.include "globals.h.s"
.include "entities/hero.h.s"
.include "collisions/collisionsprofile.h.s"
.include "gamemanager/mapmanager.h.s"

;;===========================================================================================================================================================
;; REQUIREMENT:
;; Number of enemies of this entity should be something set on stone and fixed, since we are going to play directly with the mem addresses we have available
;; If the mem addresses reserved for this enemy get stepped by something else the program will work erraticly and will fail. 
;;===========================================================================================================================================================


;; In this case we can have maximum OF entities of this enemy in the same map, so we will have to keep TBD bytes
;; CHAR SIZE 						= 		15 bytes
;; EXTRA BYTES FOR THIS ENTITY 		= 		0n bytes
charjp_num_ent: 		.db 00
defineNCharJumperDefault charjp_vector


;;=============================================================
;; Executes (HL) code for all the jumpers
;; Inputs: 
;; 		HL => Pointer to the routine to execute
;;=============================================================
charjp_doForAll::
	ld a, (charjp_num_ent) 		
	or a 									;; | if there are no entities spawned we simply return
	ret z 									;; |
	ld iy, #CHAR_JUMPER_ADDRESS 
	ld (charjp_rountine_to_call), hl
	
	;; Warning: if weird behaviours, please track the content of iy, it might be modified on a subroutine
	;; called by the self modified code

	charjp_doThings_loop:
		push af 							;; save af on the stack FIXME: check if it can be done with ex af'
		charjp_rountine_to_call = . + 1 	;; Auto modificable code, hl will store the function we'll call
		call ru_drawMaskedEntity			;; draws iy entity 
		pop af 								;; restore af FIXME: check if it can be done with ex af'
		ld bc, #CHAR_JUMPER_SIZE 			;; number of bytes to jump
		add iy, bc 							;; iy stores the new mem address after the jump
		dec a
	jr nz, charjp_doThings_loop

ret


;;=============================================================
;; Adds one char_Jumper to the end of the array of char_jumpers
;; Inputs: (Spawn parameters)
;; 		D => X position where to spawn it
;;  	E => Y position where to spawn it
;;=============================================================
charjp_spawnJumper::
	ld a, (charjp_num_ent) 				;; A = number os charjp entities spawned
	cp #CHAR_JUMPER_MAX_ENT 			;; if charjp_num_ent == CHAR_JUMPER_MAX_ENT
	ret z 								;; we don't spawn one entity

	inc a 								;; If we are able to spawn an entity then
	ld (charjp_num_ent), a 				;; we update the current number of entities we have spawned

	dec a 				 				;; we want current number - 1 since its an array getter
	call charjp_getJumperById			;; Based on the num_ent, we resolve the mem address we have to write at
	ld Ent_x(iy),  d 					;; Setting defaults
	ld Ent_px(iy), d 					;; |
	ld Ent_ppx(iy),d
	ld Ent_y(iy),  e 					;; |
	ld Ent_py(iy), e
	ld Ent_ppy(iy),e 					;; |
ret


;;================================================================================
;; Gets the mem address of a char jumper giving an id
;; This function does not check boundaries
;; Inputs: (Spawn parameters)
;; 		A  => id of jumper to get (from 0 to n-1)
;; Return:
;;		IY => Returns the memory address to the first byte of the requested jumper
;; Destroys:
;; 		AF, BC, IY
;;================================================================================
charjp_getJumperById::
	ld iy, #CHAR_JUMPER_ADDRESS 		;; HL holds the first mem address
	or a 								;; | If the id is 0 we can return directly on hl the first mem address
	ret z

	ld bc, #CHAR_JUMPER_SIZE 			;; a stores the size we want to jump between mem addresses

	charjp_resolve_mem_address: 		
		add iy, bc 						;; Keeping the result on hl will make sure that in the last iteration of this loop hl will hold the correct reference if the constants are set correctly
		dec a                           ;; |
	jr nz, charjp_resolve_mem_address
ret

;;=============================================================
;; Destroys the char_jumper given a mem address
;; Inputs:
;; 		IY => Mem address of charjp to destroy
;;=============================================================
charjp_destroyJumperIY::
	ld__d_iyh
	ld__e_iyl
	jr charjp_destroyJumper

;;=============================================================
;; Destroys the char_jumper given a mem address
;; Inputs:
;; 		DE => Mem address of charjp to destroy
;;=============================================================
charjp_destroyJumper::
	ld a, (charjp_num_ent) 				;; A = number os charjp entities spawned
	or a 								;; if charjp_num_ent == 0
	ret z 								;; if there are no entities we cannot destroy the entity
	
	;;
	;; Specifics of this function: 
	;; - Erases always the last element on the imaginary array of charjp data
	;; - In order to erase a concrete element, it is needed first to do the following operations:
	;;		1. Copy the last element of the array in the mem address we want to erase
	;; 			aa, bb, cc, dd => aa, dd, cc, dd
	;;		2. Erase the last element of the array
	;; 			aa, dd, cc, dd => aa, dd, cc, xx
	;; - So it is needed to know precisely the size of the content to be able to copy it around mem addresses without corrupting anything
	;;

	;;========================================================================
	;; Copy the last element of the array in the mem address we want to erase
	;;========================================================================
	ld a, (charjp_num_ent) 		;; a == Current number of entities, the previous check ensures it's over 0
	dec a 						;; we want current number - 1 since its an array getter
	call charjp_getJumperById  	;; By calling this function we have on iy the memory address we want to copy
	ld__d_iyh 					;; store in d, iyh which is the high part of the mem address we want to copy
	ld__e_iyl 				 	;; store in e, iyl which is the low part of the mem address we want to copy
	ex de, hl  					;; We store for ldir the content of iy on hl
	ld bc, #CHAR_JUMPER_SIZE  	;; bc stores the 'class' size to transfer it later on with ldir
	ldir 						;; we copy hl on de, this means that the step one is completed, see previous comment

	;;========================================================================
	;; Erase the last element of the array
	;;========================================================================
	ld a, (charjp_num_ent) 		;; First, we decrease the number of elements we'll track
	dec a 						;; |
	ld (charjp_num_ent), a 		;; |

	;; We could zero out the last element of the array, but that will take out some cycles and bytes we need for other operations
	;; So by design, we are going to leave "dead data" in the last position of that memory address, the dead data will be 
	;; replaced by fresh data every time we will spawn new elements

ret


;;=============================================================
;; 'Destroys' all the char_jumpers
;; 	DESTROY: AF
;;=============================================================
charjp_destroyAllJumpers::
	xor a
	ld (charjp_num_ent), a
	ret

;;=============================================================
;;	INPUTS:
;;		IY: charjp addr
;;		HL: new XY position
;;		 H: X coordinate
;;		 L: Y coordinate
;;=============================================================
charjp_moveToXY::
	ld Ent_x(iy), h
	ld Ent_y(iy), l
	ret

;;=============================================================
;;	Function that manages the correct movement to the right.
;;	INPUTS:
;;		IY: charjp addr
;;	DESTROYS: A
;;=============================================================
charjp_moveRight::
	ld hl, #charjp_response_to_tiles		;; Response to tiles
    ld bc, #char_changeLookSideHorizontally	;; If hit a wall, the looking side will swap
	jp char_moveRight						;; Move right

;;=============================================================
;;	Function that manages the correct movement to the left.
;;	INPUTS:
;;		IY: charjp addr
;;	DESTROYS: A
;;=============================================================
charjp_moveLeft::
	ld hl, #charjp_response_to_tiles		;; Response to tiles
    ld bc, #char_changeLookSideHorizontally	;; If hit a wall, the looking side will swap
	jp char_moveLeft						;; Move left

;;=============================================================
;;	Function that kills a charjp.
;;	INPUTS:
;;		IY: charjp addr
;;=============================================================
charjp_ondeath::
	ld__d_iyh					;; |
	ld__e_iyl					;; |
	call charjp_destroyJumper	;; \ Current charjp is death

	push iy
	call mm_drawTiles
	pop iy
	call ru_switchBuffer     	;; Wait for the raster to trigger vsync
	push iy
	call mm_drawTiles
	pop iy
	call ru_switchBuffer     	;; Wait for the raster to trigger vsync

	ld a, #0xFF					;; Signal to end
	ret



;;===============================================================
;;	Behaviour when a charjp hit specifics tiles.
;;	INPUTS:
;;		IY: charjp addr
;;===============================================================
charjp_response_to_tiles:
    ld a, h									;; |
    cp #RANGE_DEATHLY						;; |
	jr z, charjp_range_change_looking_side	;; \ If hit tile is a deathly tile, jr charjp_range_change_looking_side
	ld a, l									;; |
    cp #RANGE_DEATHLY						;; |
	jr z, charjp_range_change_looking_side	;; \ If hit tile is a deathly tile, jr charjp_range_change_looking_side
	ld a, h									;; |
    cp #RANGE_NEXT_MAP_SAME_LVL				;; |
	jr z, charjp_range_change_looking_side	;; \ If hit tile is a tile to go to the next level, jr charjp_range_change_looking_side
	ld a, l									;; |
    cp #RANGE_NEXT_MAP_SAME_LVL				;; |
	jr z, charjp_range_change_looking_side	;; \ If hit tile is a tile to go to the next level, jr charjp_range_change_looking_side
	ret
	
	charjp_range_change_looking_side:

	ld hl, #returnToPreviousFunction		;; It avoids to restore the looking side of the function that is being executed (char_moveLeft or char_moveRight)
											;; When char_moveLeft or char_moveRight is being executed, they execute the function that is put in HL to check tiles (i.e. this)
											;;	but when finish the execution, the function char_moveLeft or char_moveRight continue the execution and they smash Char_look_side
											;;	so if we stop that execution, the smashing will not be executed, and we got that going back and back again, working on stack.
											;;	returnToPreviousFunction is what actually does. If we "call" returnToPreviousFunction just will go back 1 function, but
											;;	this function is being embedded, so we need 2 steps back, and doing "jp" is what actually got! Sweet.
	push hl
	jp char_changeLookSideHorizontally		;; Swap looking side

;;=============================================================
;;	Behaviour when a charjp hit specifics tiles. Specifically,
;;		charjp will die when hit deathly tiles.
;;	INPUTS:
;;		IY: charjp addr
;;		 H:  left-bottom point ID tile
;;		 L: right-bottom point ID tile
;;=============================================================
charjp_response_to_tiles_death:
    ld a, h					;; |
    cp #RANGE_DEATHLY		;; |
	jr z, charjp_ondeath	;; \ If hit tile is a deathly tile, charjp dies
	ld a, l					;; |
    cp #RANGE_DEATHLY		;; |
	jr z, charjp_ondeath	;; \ If hit tile is a deathly tile, charjp dies
	ret

;;=============================================================
;;	Manages the gravity of the charjp
;;	INPUTS:
;;		IY: charjp addr
;;	DESTROYS: 
;;=============================================================
charjp_gravity::
    ld hl, #charjp_response_to_tiles_death	;; Tiles management
    ld de, #charjp_ondeath					;; If charjp hit the bottom boundaries, it will die
	
	jp char_gravity						;; Manage character's gravity

;;=============================================================
;;	It swaps the movement between left and right
;;	INPUTS:
;;		IY: charjp addr
;;	DESTROYS: 
;;=============================================================
charjp_moveRightAndLeftIfOnGround:
	ld a, Char_state(iy)		;; A = charjp's state
	cp #CHAR_FALLING			;; |
	ret z						;; \ If charjp is falling, ret
	ld a, Char_look_side(iy)	;; A = charjp looking side
	cp #ANIM_Normal_right		;; |
	jp z, charjp_moveRight		;; \ If charjp is looking at the right, go right
	cp #ANIM_Normal_left		;; |
	jp z, charjp_moveLeft		;; \ If charjp is looking at the left,  go left
	ret

;;=============================================
;;	Starts the jump of the charjp.
;;	INPUTS:
;;		IY: charjp addr
;; 	DESTROYS: 
;;=============================================
charjp_startJump:
	ld a, Char_state(iy)	;; A = charjp's state
	cp #CHAR_JUMPING		;; |
	ret z					;; \ If charjp is jumping, ret
	cp #CHAR_FALLING		;; |
	ret z					;; \ If charjp is falling, ret

	ld hl, #doNothing		;; Function to manage the wall jumping
	jp char_startJump		;; Start the jump, but not jumping yet

;;=============================================
;;	It does the charjp jump.
;;	INPUTS:
;;		IY: charjp addr
;; 	DESTROYS: 
;;=============================================
charjp_jumpControl:
	ld hl, #doNothing		;; Function to manage the wall jumping
	ld de, #charjp_ondeath
	jp char_jumpControl		;; When jumping is started, then can be controlled and the charjp jumps

;;=============================================
;;	Returns if charjp is seeing the hero
;;	INPUTS:
;;		HL: charjp's XY in tile coordinates
;;		DE: hero's XY in tile coodinates
;;		 B: max tile distance
;;	OUTPUTS:
;;		 A: A = 1 if seeing hero, A = 0 if not
;; 	DESTROYS: 
;;
;;---------------------------------------------
;;
;;	If INPUT A = 1:
;;
;;	-------------
;;	|   |   |   |
;;  -------------
;;  |   | C |   |
;;  -------------
;;  |   |   |   |
;;  -------------
;;
;;	C: charjp in middle tile.
;;	If hero in some, returns that is seeing
;;		the hero.
;;
;;=============================================
charjp_X_minus_hero_X_in_tile: .db #0x00	;; charjp's X coordinate - hero's X coordinate
charjp_Y_minus_hero_Y_in_tile: .db #0x00	;; charjp's Y coordinate - hero's Y coordinate

charjp_isCharjpSeeingHero::
	ld a, h									;; |
	sub d									;; \ A = H - D = charjp's X coordinate - hero's X coordinate

	ld (charjp_X_minus_hero_X_in_tile), a	;; Saving the value, so will not to recalculate later

	push af									;; Saving A = charjp's X coordinate - hero's X coordinate

	ld a, l									;; |
	sub e 									;; \ A = L - E = charjp's Y coordinate - hero's Y coordinate

	ld (charjp_Y_minus_hero_Y_in_tile), a	;; Saving the value, so will not to recalculate later

	call absValue							;; A = |charjp's Y coordinate - hero's Y coordinate|
	ld h, b									;; H = B = max tile distance
	ld l, a									;; L = A = |charjp's Y coordinate - hero's Y coordinate|
	call greaterOrEqual						;; A = 1 if H >= L | A = 0 if H < L
	or #0									;; |
	jr z, charjp_not_seeing_hero_p			;; \ Charjp is not seeing the hero

	pop af									;; A = charjp's X coordinate - hero's X coordinate

	call absValue							;; A = |charjp's X coordinate - hero's X coordinate|
	ld h, b									;; H = B = max tile distance
	ld l, a									;; L = A = |charjp's X coordinate - hero's X coordinate|
	call greaterOrEqual						;; A = 1 if H >= L | A = 0 if H < L
	or #0									;; |
	jr z, charjp_not_seeing_hero			;; \ Charjp is not seeing the hero

	ld a, #1								;; Charjp is seeing the hero
	ret

	charjp_not_seeing_hero_p:
		pop af								;; Necessary pop
	charjp_not_seeing_hero:
		xor a								;; Charjp not seeing the hero

	ret

;;=============================================
;;	Function to resume the calls to perform 
;;		a jump on the charjp.
;;	INPUTS:
;;		IY: charjp addr
;;=============================================
charjp_doJump:
	call charjp_startJump	;; Starts   the jump
	jr charjp_jumpControl	;; Performs the jump

;;=============================================
;;	It kills the hero.
;;=============================================
charjp_killHero:
	jp hero_kill	;; It kills the hero

;;=============================================
;;	It checks if the hero and the charjp are
;;		colliding and if they are doing it,
;;		hero will die.
;;	INPUTS:
;;		IY: charjp addr
;;	OUTPUTS:
;;		 A: 0 if not colliding, 1 if colliding
;;=============================================
charjp_collidingWithHero:
	;call hero_getPtrHL						;; HL = hero's data
	;call isColliding						;; It checks if the hero and the charjp are colliding
	;ret

	ld hl, #isColliding
	push hl
	jp hero_getPtrHL

	;; call addrA + call addrB + ... + call addrN = ld hl, #NEXT_ADDR_INST_AFTER_JP + push hl + ld hl, addrN + push hl + ... + ld hl, addrB + push hl + jp addrA
	;;
	;; 1: n times     call addrX     ( n * 7B      | n * 17C       | min. 2 * n stack movements)
	;; 2.1: n - 1 times ld hl, addrX ( n * 3B      | n * 10C       |          0 stack movements)
	;; 2.2: n - 1 times push hl      ( n * 1B      | n * 11C       | min.     n stack movement)
	;; 2.3: jp addrX                 (     3B      |     10C       |          0 stack movements)
	;; 2: 							 ( n * 4B + 3B | n * 21C + 10C | min.     n stack movements)
	;;
	;; 1 vs 2: 2 consumes 7/4 times less space. 2 consumes 21/17 times more time. 2 makes n stack movements less.
	;;
	;; n = 1
	;; 1: ( 7B | 17C | min. 2 stack movements)
	;; 2: ( 7B | 31C | min. 1 stack movement)
	;; n = 2
	;; 1: (14B | 34C | min. 4 stack movements)
	;; 2: (11B | 52C | min. 2 stack movements)
	;; n = 3
	;; 1: (21B | 51C | min. 6 stack movements)
	;; 2: (15B | 73C | min. 3 stack movements)
	;; n = 4
	;; 1: (28B | 68C | min. 8 stack movements)
	;; 2: (19B | 94C | min. 4 stack movements)

	;; call addrA + call addrB + ret = ld hl, addrB + push hl + jp addrA
	;;
	;; 1: call addrA   (3B | 17C | min. 2 stack movements) + call addrB (3B | 17C | min. 2 stack movements) + ret      (1B | 10C | 1 stack movement)
	;; 2: ld hl, addrB (3B | 10C | 0 stack movements)      + push hl    (1B | 11C | 1 stack movement)       + jp addrA (3B | 10C | min. 1 stack movements)
	;;
	;; 1: 7B | 44C | min. 5 stack movements
	;; 2: 7B | 31C | min. 2 stack movements

	;; call addrA + call addrB + ... + call addrN + ret = ld hl, addrN + push hl + ... + ld hl, addrB + push hl + jp addrA
	;;
	;; 1.1: n times call addrX (n * 3B | n * 17C | min. 2 * n stack movements)
	;; 1.2: ret (1B | 10C | 1 stack movement)
	;; 1: (n * 3B + 1 | n * 17C + 10C | min. 2 * n + 1 stack movements)
	;; 2.1: n - 1 times ld hl, addrX ((n - 1) * 3B | (n - 1) * 10C | 0 stack movements)
	;; 2.2: n - 1 times push hl ((n - 1) * 1B | (n - 1) * 11C | min. n stack movements)
	;; 2.3: jp addrA (3B | 10C | min. 1 stack movements)
	;; 2:   ((n - 1) * 4B + 3B | (n - 1) * 21C + 10C | min. n stack movements)
	;; 2:   (n * 4B - 1B | n * 21C - 11C | min. n stack movements)
	;;
	;; 1 vs 2: 1 consumes 4/3 times less space. 2 consumes 21/17 times more time. 2 makes n - 1 stack movements less.
	;;
	;; n = 2
	;; 1: ( 7B | 44C | min. 5 stack movements)
	;; 2: ( 7B | 31C | min. 2 stack movements)
	;; n = 3
	;; 1: (10B | 61C | min. 7 stack movements)
	;; 2: (11B | 52C | min. 3 stack movements)
	;; n = 4
	;; 1: (13B | 78C | min. 9 stack movements)
	;; 2: (15B | 73C | min. 4 stack movements)
	;; n = 5
	;; 1: (16B | 95C | min. 11 stack movements)
	;; 2: (19B | 94C | min. 5 stack movements)
	;; n = 6
	;; 1: (19B | 112C | min. 13 stack movements)
	;; 2: (23B | 115C | min. 6 stack movements)

;;=============================================
;;	If some code is needed to finish the 
;;		the normal execution of normal
;;		behaviour (i.e. charjp not seeing hero)
;;		it will be here (e.g. finish a jump).
;;	INPUTS:
;;		IY: charjp addr
;;=============================================
charjp_finishBehaviourWhenNotSeeingHero:
	ld a, Char_state(iy)						;; A = charjp addr
	cp #CHAR_IDLE								;; |
	ret z										;; \ If hero is not moving, ret
	cp #CHAR_WALKING							;; |
	ret z										;; \ If hero is walking, ret
	jr charjp_behaviourNotSeeingHeroNoSprite	;; If hero is falling or jumping, finish it

;;=============================================
;;	Normal behaviour (i.e. when not seeing the 
;;		hero).
;;	INPUTS:
;;		IY: charjp addr
;;=============================================
charjp_behaviourNotSeeingHero:
	ld Ent_spr_l(iy), #_charjumper_sprite		;; Loads normal sprite on charjp

	charjp_behaviourNotSeeingHeroNoSprite:
	call charjp_moveRightAndLeftIfOnGround		;; Alternate movement between left and right

	jr charjp_doJump							;; It makes the charjp jump

;;=============================================
;;	If charjp is seeing the hero, this function
;;		will be executed.
;;	INPUTS:
;;		IY: charjp addr
;;=============================================
charjp_behaviourSeeingHero:
	ld Ent_spr_l(iy), #_charjumper_angry_sprite		;; Loads angry sprite on charjp

	call charjp_finishBehaviourWhenNotSeeingHero	;; It finish the jump of the charjp

	ld a, Char_state(iy)							;; A = charjp's state
	cp #CHAR_JUMPING								;; |
	ret z											;; \ If charjp is jumping, ret
	cp #CHAR_FALLING								;; |
	ret z											;; \ If charjp is falling, ret

	ld a, (charjp_X_minus_hero_X_in_tile)			;; Loading (charjp's X coordinate - hero's X coordinate)

	call greaterOrEqualThanZero						;; Checks if A >= 0
	or a											;; Same that cp #0
	push af											;; Saving flags
	call  z, charjp_moveRight						;; If A < 0 -> hero is at the right of charjp
	pop af											;; Restoring flags
	call nz, charjp_moveLeft						;; If A > 0 -> hero is at the left of charjp

	ld a, (charjp_Y_minus_hero_Y_in_tile)			;; Loading (charjp's Y coordinate - hero's Y coordinate)
	dec a											;; Dec 1 unit to avoid the charjp being jumping if hero and charjp have the same Y coordinate (same height). Doing this, charjp follows the hero walking, not jumping
	call greaterOrEqualThanZero						;; Cheks if A >= 0
	or a											;; Same that cp #0
	call nz, charjp_doJump							;; If A > 0 -> hero is above the charjp, so charjp have to jump to reach the hero

	ret

;;=============================================
;;	AI for the charjp.
;;	Simple AI: state machine
;;	INPUTS:
;;		IY: charjp addr
;;=============================================

charjp_ai::
	call hero_getXYPosition						;; HL = hero's XY coordinates

	call mm_ConvertToTileCoordinates			;; HL = hero's XY position in tile coordinates
	push hl
	
	ld h, Ent_x(iy)								;; |
	ld l, Ent_y(iy)								;; \ HL = charjp's XY coordinates

	call mm_ConvertToTileCoordinates			;; HL = charjp's XY position in tile coordinates
	pop de										;; DE = hero's XY position in tile coordinates

	ld b, #3
	call charjp_isCharjpSeeingHero				;; A = 1 if seeing hero, A = 0 if not
	or a										;; Same that cp #0x00
	jr z, notSeeingHero							;; If A  = 0, jr notSeeingHero
	jr nz, seeingHero							;; If A != 0, jr seeingHero

	notSeeingHero:
		jr charjp_behaviourNotSeeingHero		;; Normal behaviour (i.e. not seeing the hero)
												;; If not seeing hero, not necessary other checks

	seeingHero:
		call charjp_behaviourSeeingHero			;; Aggressive behaviour (i.e. seeing the hero)
	
		call charjp_collidingWithHero			;; Checks if charjp is touching the hero
		or a									;; Same that cp #0
		call nz, charjp_killHero				;; If A != 0 it means that charjp is touching the hero, so charjp have to kill the hero

	ret

;;=============================================
;; 	Update all the charjp.
;; 	DESTROYS: 
;;=============================================
charjp_tick: .db #CHAR_JUMPER_TICKS		;; To decrease the number of operations, charjp's operations will be executed 1/CHAR_JUMPER_TICKS per call to charjp_updateForAll
										;; Example: if CHAR_JUMPER_TICKS = 3, to execute charjp_updateForAll, this function will have to be called 3 times, and so on

charjp_updateForAll::

	ld a, (charjp_tick)					;; A = current tick
	or a								;; Same that cp #0
	jp z, charjp_continue_updateForAll	;; If tick = 0, update for all charjp
	dec a								;; If tick != 0, dec tick
	ld (charjp_tick), a					;; Save the new tick value
	ret

	charjp_continue_updateForAll:
	ld a, #CHAR_JUMPER_TICKS			;; A = initial number of ticks
	ld (charjp_tick), a					;; Restore initial number of ticks
	
	ld hl, #charjp_gravity				;; |
	call charjp_doForAll				;; \ Manage gravity for all charjp

	ld hl, #charjp_ai					;; |
	jp charjp_doForAll					;; \ Manage AI for all charjp
