;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.area _DATA
.area _CODE

.include "collisions/collisionsprofile.h.s"
.include "collisions/collisions.h.s"
.include "gameplaystatics.h.s"
.include "entity.h.s"
.include "globals.h.s"
.include "gamemanager/mapmanager.h.s"
.include "renderutils.h.s"
.include "entities/character.h.s"

.include "entities/hero.h.s"


;;=============================================
;;  Jump Table
;;=============================================
jumptable: 	
    .db #05, #04, #03, #02
	.db #01, #01, #01, #00 
    .db #00,  #00, #0x80

;;=============================================
;;=============================================
;; PUBLIC FUNCTIONS
;;=============================================
;;=============================================

;;=============================================
;; Common collision checks for right left movement
;; DESTROYS: AF, BC
;; INPUT: 
;; 		l => lower tile
;; 		h => higher tile
;;=============================================
char_rightleftbottom_collision_checks:
	;; At this point we have on b the higher tile and on a the lower tile
    ld a, l
    cp #RANGE_BLOCK_ALL      ;; if lower_tile = RANGE_BLOCK_ALL Set the z flag and block movement
    ret z
    ld a, h                  ;; a now is equal to the higher tile range
    cp #RANGE_BLOCK_ALL      ;; if higher_tile = RANGE_BLOCK_ALL Set the z flag and block movement
    ret z

        ;; Else special conditions
    ret

;;=============================================
;; It determines what to do when a right/left collision with a specific ID happen
;; It sets the flag zero if something solid is blocking the path of the character
;; It will execute arbitrary code as needed otherwise
;; DESTROYS: AF, BC, HL, DE
;; INPUT: 
;;	IY => character_data
;; 	HL => Points to the function we want to evaluate/boundary 
;;=============================================
char_ProcessRightLeftGroundCollision::
    ld (proc_col), hl   		;; hl points to the function to check tiles			
	
    ld b, #01 					;; b is the offset
    
    ;proc_col = . + 1 			;; proc_col will modify the line we have down there from right to left
    ;call entityIsCollidingWithTileRight ;; Are we colliding with something solid on the left
    ;call whichRangeIDIs2Tiles
    ;call char_rightleftbottom_collision_checks  ;; Common collision check for right-left-bottom cases
    ;ret

    ;call entityIsCollidingWithTileRight ;; Are we colliding with something solid on the left
    ;call whichRangeIDIs2Tiles
    ;jp char_rightleftbottom_collision_checks  ;; Common collision check for right-left-bottom cases

    ;call entityIsCollidingWithTileRight ;; Are we colliding with something solid on the left
    ;ld de, #char_rightleftbottom_collision_checks
    ;push de
    ;jp whichRangeIDIs2Tiles  ;; Common collision check for right-left-bottom cases

    ld de, #char_rightleftbottom_collision_checks
    push de
    ld de, #whichRangeIDIs2Tiles
    push de
    proc_col = . + 1 			        ;; proc_col will modify the line we have down there from right to left
    jp entityIsCollidingWithTileRight


;;=============================================
;;  This function updates (x, px, ppx) and (y, py, ppy)
;;  INPUTS:
;;      IY: character addr
;;  DESTROYS: AF
;;=============================================
char_update_positions::
    ld a, Ent_px(iy)
    ld Ent_ppx(iy), a
    ld a, Ent_x(iy)
    ld Ent_px(iy), a

    ld a, Ent_py(iy)
    ld Ent_ppy(iy), a
    ld a, Ent_y(iy)
    ld Ent_py(iy), a

    ld a, Ent_draw(iy)
    dec a
    ld Ent_draw(iy), a

    ret

;;=============================================
;;  Move character left if is not at the limit
;;  INPUTS:
;;      IY: character addr
;;      HL: function addr to manage tiles collision behaviour
;;      BC: function addr to manage blocking tiles
;;  DESTROYS: AF, HL
;;=============================================
char_moveRight::
    ld  a, Ent_x(iy)              ;; Store hero_x in a 
    ld  e, a                     ;; Store hero_x in e 
    ld  a, Ent_w(iy)              ;; Store hero_w in a 
    add e                        ;; a = hero_w + hero_x
    cp  #MAP_END_X               ;; Check against right limit ((hero_w + hero_x) == screen_limit)  
    ret z


    ld (response_to_tiles_right), hl
    ld (blocking_tiles_right), bc
    ld hl, #entityIsCollidingWithTileRight
    push iy
    call char_ProcessRightLeftGroundCollision
    pop iy
    push af
    ;jr z, do_not_move_r   ;; Hero_X == Limit, do not move right
    blocking_tiles_right = . + 1
    ;call z, returnToPreviousFunction
    call z, doNothing
    pop af
    ret z

    response_to_tiles_right = . + 1
    call doNothing
   
    ld a, Ent_x(iy)
    ;; Move Right
    inc a                       ;; hero_x++
    ld Ent_x(iy), a              ;; Update Hero_x 
    ld a, #2
    ld Ent_draw(iy), a 
    ld a,  #ANIM_Normal_right   ;; |
    ld Char_look_side(iy), a      ;; | Hero is looking to the right
    
    ld a, Char_state(iy)           ;; If we are falling we don't change the state to walking
    cp #CHAR_FALLING
    ret z
    cp #CHAR_JUMPING
    ret z

    ld a, #CHAR_WALKING         ;; 
    ld Char_state(iy), a          ;; | Hero state is walking now
        
    ;do_not_move_r:        ;; jr is better than jp, however jp jumps longer in code
    ret

;;=============================================
;;  Move character right if is not at the limit
;;  INPUTS:
;;      IY: character addr
;;      HL: function addr to manage tiles collision behaviour
;;      BC: function addr to manage blocking tiles
;;  DESTROYS: AF, HL
;;=============================================
char_moveLeft::
    ld a, Ent_x(iy)
    cp  #MAP_ORIGIN_X     ;; Check against left limit
    ;jr z, do_not_move_l   ;; Hero_X == Limit, do not move right 
    ret z

    ld (response_to_tiles_left), hl
    ld (blocking_tiles_left), bc
    ld hl, #entityIsCollidingWithTileLeft
    push iy
    call char_ProcessRightLeftGroundCollision
    pop iy
    push af
    ;jr z, do_not_move_l   ;; If it's a blocking tile then don't move
    blocking_tiles_left = . + 1
    ;call z, returnToPreviousFunction
    call z, doNothing
    pop af
    ret z

    response_to_tiles_left = . + 1
    call doNothing

    ld a, Ent_x(iy)
    ;; Move left
    dec a             ;; hero_x--
    ld Ent_x(iy), a
    ld a, #2
    ld Ent_draw(iy), a
    ld a,  #ANIM_Normal_left   ;; |
    ld Char_look_side(iy), a
    
    ld a, Char_state(iy)
    cp #CHAR_FALLING
    ret z
    cp #CHAR_JUMPING
    ret z
    
    ld a, #CHAR_WALKING         ;; 
    ld Char_state(iy), a

        
    ;do_not_move_l:        ;; jr is better than jp, however jp jumps longer in code
    ret

;;=============================================
;;  This function erases and draws a character
;;  INPUTS:
;;      IY: character addr
;;  DESTROYS: AF, BC, HL, DE, IX, IY
;;=============================================
char_changeLookSideHorizontally::
    ld a, Char_look_side(iy)
    cp #ANIM_Normal_right
    jr z, char_changeLookSideToLeft
    
    ; Change look side to right
    ld a, #ANIM_Normal_right
    ld Char_look_side(iy), a
    ret

    ; Change look side to left
    char_changeLookSideToLeft:
        ld a, #ANIM_Normal_left
        ld Char_look_side(iy), a
    ret

;;=============================================
;;  Controls Jump movements
;;  INPUTS:
;;      IY: char addr
;;      HL: function addr to response to tails
;;      DE: function addr to kill entity
;;  DESTROYS: AF, BC, HL, DE
;;=============================================
char_jumpControl::

    ld (char_jumpControl_response_to_tiles_inv), hl
    ld (char_jumpControl_response_to_tiles_normal), hl
    ld (char_kill_entity_inv), de
    ld (char_kill_entity_normal), de

    ;; Check if we are jumping right now
    ;ld      a, (hero_jump_index)        ;; A = hero_jump_index status
    ld      a, Char_jmp(iy)
    cp      #-1                         ;; A == -1? (-1 is not jumping)
    ret     z                           ;; If A == -1, not jumping

    ld  hl, #jumptable  ; HL = hero jump table ground pointer
    ;; Get Jump Value
    ;ld       a, (hero_jump_index)
    ld       a, Char_jmp(iy)
    ld       c, a               ;; |
    ld       b, #0              ;; | BC = A (Offset)
    add     hl, bc              ;; HL += BC

    ;; Check End of jumping
    ld      a, (hl)                     ;; A = Jump movement
    cp      #0x80                       ;; Jump Value == 0x80? End of jump
    jr      z, char_jump_end
    or      a                           ;; If jump value == 00
    jr      z, char_refresh_jump_index  ;; Then we can go directly to refresh the jump index

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;; JUMP STILL GOES
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;; Do Jump Movement, based on the gravity we have to check collision up or down
    ld e, a                         ;; | We store in e the jump movement we need to proceed to make
    ;ld a, (hero_gravdirection)      ;; |
    ld a, Char_grav_dir(iy)         ;; |
    cp #CHAR_GRAV_NORMAL            ;; | If we are in normal gravity
    jr z, char_normal_gravity_jump

        ;; Inverse gravity jump
        char_inv_grav_jump_loop:
            push de                                     ;; pushing de on the stack
            push iy
            ld hl, #entityIsCollidingWithTileBottom     ;; Checking collisions with tile bottom since its an inverted gravity jump
            call char_ProcessRightLeftGroundCollision   ;; FIXME: Consider special cubes such as the trenspasing one
            pop iy
            pop de                                      ;; restoring de
            jr z, char_refresh_jump_index               ;; TBD: Change to jump_end if faster jump is desired on streched areas
            char_jumpControl_response_to_tiles_inv = . + 1
            call doNothing
            cp #0xFF
            ret z
            ;ld a, Ent_y(iy)

            ld a, #MAP_END_Y
            dec a
            sub Ent_h(iy)
            ld b, a

            ld a, Ent_y(iy)

            cp b
            char_kill_entity_inv = . + 1
            jp z, doNothing                             ;; Kill entity

            inc a                                       ;; Pushing one unit the hero down
            ld Ent_y(iy), a                             ;; Refreshing the hero position
            ld a, #2
            ld Ent_draw(iy), a
            dec e                                       ;; Loop iteration
        jr nz, char_inv_grav_jump_loop
        jr char_refresh_jump_index


        char_normal_gravity_jump:
            push de                                     ;; pushing de on the stack
            push iy
            ld hl, #entityIsCollidingWithTileUpper      ;; Checking collisions with tile upper
            call char_ProcessRightLeftGroundCollision   ;; FIXME: Consider special cubes such as the trenspasing one
            pop iy
            pop de                                      ;; restoring de
            jr z, char_refresh_jump_index               ;; TBD: Change to jump_end if faster jump is desired on streched areas
            char_jumpControl_response_to_tiles_normal = . + 1
            call doNothing
            cp #0xFF
            ret z
            ld a, Ent_y(iy)

            ld b, #MAP_ORIGIN_Y
            inc b
            cp b
            char_kill_entity_normal = . + 1
            jp z, doNothing             ;; Kill entity

            dec a                                       ;; Pushing one unit the hero down
            ld Ent_y(iy), a                             ;; Refreshing the hero position
            ld a, #2
            ld Ent_draw(iy), a
            dec e                                       ;; Loop iteration
        jr nz, char_normal_gravity_jump


        ;; Increment hero_jump_index Index
        char_refresh_jump_index:
        ld       a, Char_jmp(iy)        ;; A = hero_jump_index
        inc      a                      ;; |
        ld Char_jmp(iy), a              ;; | hero_jump_index++
        ret

    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;; JUMP ENDED
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    char_jump_end:
    ;; Put -1 in the jump index when jump ends
    ld      a, #-1                  ;; |
    ld      Char_jmp(iy), a         ;; | hero_jump_index = -1
    ld      a, #CHAR_FALLING
    ld      Char_state(iy), a
    ret

;;=============================================
;;  
;;  INPUTS:
;;      IY: character addr
;;      HL: function to call to check wall jump
;;=============================================
char_startJump::

    ld (startJump_function), hl

    ;ld a, (hero_state)  ;; If we are falling, we shouldn't let the player jump
    ld a, Char_state(iy)
    cp #CHAR_JUMPING
        ret z
    cp #CHAR_FALLING
        ;jr z, checkWallJump
        startJump_function = . + 1
        jp z, doNothing
    

    ;; Jump inactive, you can start jumping
    ld  a, #0                   ;; |
    ;ld  (hero_jump_index), a    ;; | hero_jump_index = 0 (Activate jumping)
    ld Char_jmp(iy), a
    ld a, #CHAR_JUMPING         ;; Refreshing hero state so now character is jumping up
    ;ld (hero_state), a          ;; |
    ld Char_state(iy), a

    ret

;;==========================================================
;;  INPUTS:
;;      HL: function addr to response to tails
;;      DE: function addr to response to floor out of bounds
;;      IY: character addr
;;==========================================================
char_gravity::

    ld a, #MAP_END_Y    ;; A = 200
    sub Ent_h(iy)       ;; A = 200 - B = 200 - sprite height
    ld (char_gravity_falling_bottom_limit), a
    ld (char_gravity_falling_response_to_tiles), hl
    ld (char_gravity_falling_out_of_bounds), de
    

    ;ld a, (hero_state)              ;; If character is idling (not moving)
    ld a, Char_state(iy)
    cp #CHAR_IDLE                   ;; we don't need to exec any kind of  gravity code
    ret z
    cp #CHAR_JUMPING
    ret z
  
    ld e, #CHARACTER_INITIAL_FALLING_SPEED  ;; Initial falling speed
    
    ;; Entering on a falling state
    ld a, #CHAR_FALLING
    ld Char_state(iy), a

    ld a, Char_grav_dir(iy)
    cp #CHAR_GRAV_NORMAL
    jr z, char_gr_fall_down

    char_gr_fall_up:
        ld hl, #entityIsCollidingWithTileUpper
        ld (char_gr_falling_collision_function), hl
        ld a, #MAP_ORIGIN_Y
        ld (char_gravity_falling_bottom_limit), a
        ld a, #0x3D
        ld (char_gravity_opcode), a                                                 ; Opcode = dec a
        jr char_gr_falling

    char_gr_fall_down:
        ;ld (char_gravity_falling_bottom_limit), #200 - CHAR_HEIGHT                 ; Default option. Is being done above
        ld hl, #entityIsCollidingWithTileBottom
        ld (char_gr_falling_collision_function), hl
        ld a, #0x3C
        ld (char_gravity_opcode), a                                                 ; Opcode = inc a


    char_gr_falling:
        push de
        push iy
        char_gr_falling_collision_function = . + 1
        ld hl, #doNothing
        call char_ProcessRightLeftGroundCollision
        pop iy
        pop de
        jr z, char_on_ground                            ;; If it's a blocking tile then don't move
        char_gravity_falling_response_to_tiles = . + 1
        call doNothing
        cp #0xFF
        ret z
        ld a, Ent_y(iy)
        char_gravity_opcode = .
        inc a                                           ;; Pushing down the character
        char_gravity_falling_bottom_limit = . + 1       ;; |
        cp #0                                           ;; \ max_bottom screen
        char_gravity_falling_out_of_bounds = . + 1      ;; |
        jp z, doNothing                                 ;; \ if hero stepped on bottom screen he falled on void
        ld Ent_y(iy), a                                 ;; Updating char position
        ld a, #2
        ld Ent_draw(iy), a
        dec e                                           ;; E--
        jr nz, char_gr_falling
    ret

    char_on_ground:
        ;; Just landed

        ld a, Char_state(iy)    ;; |
        cp #CHAR_WALKING        ;; | We check if we are in a walking state
        ret z                   ;; |
        ld a, #CHAR_IDLE        ;; | Once we touch the ground we go to an idle state if we are not walking
        ld Char_state(iy), a    ;; We refresh the character state
    ret