;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.include "gameplaystatics.h.s"
.include "collisionsprofile.h.s"
.include "globals.h.s"

range_first = .                             ;; Memory pointer to know where we start our range set

;RANGE_NO_COLLISIONS = 0                    ;; It does not generate code. Needed to know range ID
range_no_collisions:        .dw #0x000C     ;; Set of tiles IDs which entity does not collide
;RANGE_NEXT_MAP_SAME_LVL = 1                ;; It does not generate code. Needed to know range ID
range_next_map_same_lvl:    .dw #0x0D0D     ;; Set of tiles which indicates the next map transition
;RANGE_BLOCK_ALL = 2                        ;; It does not generate code. Needed to know range ID
range_block_all:            .dw #0x0E1C     ;; Set of tiles IDs which entity is blocked upper, bottom, left and right  9-22 in decimal
;RANGE_DEATHLY = 3                          ;; It does not generate code. Needed to know range ID.
range_deathly:              .dw #0x1D24     ;; Set of tiles IDs which entity dies              
;RANGE_GRAVITY = 4                          ;; It does not generate code. Needed to know range ID.
range_gravity:              .dw #0x2525     ;; Set of tiles IDs which entity changes its gravity              

range_last: .db #RANGE_ID_LAST              ;; Necessary to know that our set of ranges have finished

;;=============================================
;;  Given a range ptr it returns a range ID
;;  NOTE: range ID have to be between
;;      [0 to inf+] and ordered [0, 1, 2, ...]
;;  INPUTS:
;;      HL: range ID addr
;;  OUTPUTS:
;;       A: range ID
;;  DESTROYS: AF, HL, DE, BC
;;=============================================
transformRangePtrToRangeID:
    ;; (HL - range_first) / 2 = (HL + (-range_first)) / 2
    ;; It is divided by 2 because it is iterating over words, which in this architecture are 2 bytes
    ;; If a generic function is needed to get the differente between 2 addr, just is
    ;;   needed change that 2 by the number of bytes that are jumping

    ld d, h                 ;; |  D = H
    ld e, l                 ;; \ DE = HL
    ld hl, #range_first     ;; HL = range_first addr
    call mulNeg1Reg16       ;; HL = -HL = -range_first

    add hl, de              ;; HL = HL + DE = -range_first + range ID addr = HL input + (-range_first)

    ld a, l
    srl a                   ;; A = (range_ID_addr_input + (-range_first)) / 2

    ret

;;=============================================
;;  Given a tile ID will be returned the addr
;;  of a tile range and its range ID
;;  INPUTS:
;;       A: tile ID
;;  OUTPUTS:
;;       A: range ID
;;  DESTROYS: AF, HL, DE, BC, IY
;;=============================================
whichRangeIDIs::
    ld hl, #range_first         ;; This functions jumps addr starting in range_first
    ld iy, #range_id_mem        ;; Using iy to access to self-modified code

    jr range_loop               ;; Start jumping bytes :D

    range_id_mem = .            ;; Using self-modified code it is avoided the use of stack (expensive and sometimes unnecessary calls)
    .db 0x00                    ;; 1st byte where it will be saved A
    .db 0x00                    ;; 2nd byte where it will be saved H
    .db 0x00                    ;; 3rd byte where it will be saved L

    range_loop:
    
    ld 0(iy), a                 ;; Saving A
    ld 1(iy), h                 ;; Saving H
    ld 2(iy), l                 ;; Saving L

    ld a, (hl)                  ;; A = *HL = *range_first + 2 * n (n depends of iteracion in loop, because we read from bytes reserved above)
    cp #RANGE_ID_LAST           ;; |
    jr z, range_end             ;; \ Have finished the search without find the tile ID expecified? If then, go out

    ld a, 0(iy)                 ;; Restoring A (i.e. tile ID input)

    ld b, a                     ;; B = A = tile ID input
    ld a, (hl)                  ;; A = first byte of range (first number that should be less than the next)
    ld d, a                     ;; D = A = first byte of range
    inc hl                      ;; HL++ = next byte of range
    ld a, (hl)                  ;; A = *HL = second byte of range (second number that should be greater than previous)
    ld h, a                     ;; | H = A = second number of range
    ld l, d                     ;; | L = D = first number of range
                                ;; \ [H, L] <-> H <= L
                                ;;  | Remember: little-endian architecture!
                                ;;  \ If we save .dw 0x02009, it will
                                ;;    be saved in memory first 09 and later 02
                                ;;    so first byte will be our second number
                                ;;    and second byte will be our first number!

    call checkNumberBetweenRange        ;; A = 0 if B not in range [H, L] | A = 1 if B in range [H, L]
    or a                               ;; |
    jr z, continue_looking_for_range    ;; \ If B not in range, continue looking for
    
        ld a, 0(iy)                     ;; Restore A (i.e. tile ID input)
        ld h, 1(iy)                     ;; |
        ld l, 2(iy)                     ;; \ Restore HL (i.e. range ptr where we are currently looking for A)

        jr transformRangePtrToRangeID   ;; Transform range ID addr to its corresponding range ID

    continue_looking_for_range:
    
    ld a, 0(iy)                 ;; Restore A (i.e. tile ID input)
    ld h, 1(iy)                 ;; |
    ld l, 2(iy)                 ;; \ Restore HL (i.e. range ptr where we are currently looking for A)

    inc hl                      ;; |
    inc hl                      ;; \ tile ID input have not been found in current range ID addr so updating to next range ID addr to found in it
    jr range_loop               ;; Now that range ID addr is updated, continue looking for range ID

    ;; Not found.
    range_end:

    ld a, #RANGE_DEFAULT_VALUE_IF_NOT_FOUND     ;; Range ID have not been found in any of our defined range ID so a default value of error is returned

    ret

;;=============================================
;;  It does the same that whichRangeIDIs but
;;  resolving with 2 tile ID instead of 1
;;  INPUTS:
;;      HL: 2 tile IDs
;;       H: tile ID A
;;       L: tile ID B
;;  OUTPUTS:
;;      HL: 2 range tile IDs
;;       H: range tile ID A
;;       L: range tile ID B
;;  DESTROYS: AF, HL, DE, BC, IY, IX
;;  NOTE: If needed, change IX by IY but
;;  with care (whichRangeIDIs destroys IY)
;;=============================================

whichRangeIDIs2Tiles_smc = .
.dw 0x0000

whichRangeIDIs2Tiles::
    ld ix, #whichRangeIDIs2Tiles_smc   ;; IX = whichRangeIDIsWrapper_smc (here it will be saved HL)
    ld 0(ix), h                         ;; *(whichRangeIDIsWrapper_smc + 0) = H (Save H) = tile ID A
    ld 1(ix), l                         ;; *(whichRangeIDIsWrapper_smc + 1) = L (Save L) = tile ID B
    ld a, h                             ;; A = H = tile ID A
    call whichRangeIDIs                 ;; A = range tile ID A
    ld 0(ix), a                         ;; *(whichRangeIDIsWrapper_smc + 0) = A = range tile ID A
    ld a, 1(ix)                         ;; A = *(whichRangeIDIsWrapper_smc + 1) = tile ID B
    call whichRangeIDIs                 ;; A = range tile ID B
    ld l, a                             ;; | L = A = range tile ID B
    ld h, 0(ix)                         ;; | H = *(whichRangeIDIsWrapper_smc + 0) = range tile ID A
                                        ;; \ HL = range tile ID A and B respectively
ret