;;----------------------------------------LICENSE NOTICE----------------------------------------
;;	_________ __________ __   _______ ______ ______ ___     ___ ______ ______ _________			
;;	\______   \______   /  \ /__   __/__   __\ ____\   \   /   /  __  /__   __\   _____\		
;;	      /  /      /  / /\ \   | |     | |  | \___/    \_/    \  \_\ \  | |   \  \_____		
;;	     /  /      /  /  __  \  | |     | |  |  ___\  \ ___ /  /  ____/  | |    \_____  \		
;;	    /  /      /  /  /  \  \ | |     | |  | /___/  /     \  \  \      | |      ____\  \		
;;	   /_ /      /_ /__/    \__\|_|     |_|  /_____\__\     /__/__/      |_|     \________\		
;;
;;  Code & Gfx Copyright (C) 2018 Alvaro Jover (@vorixo), Jordi Amoros (@byFlowee) 
;;	and Cristian Garcia (@cgr71ii)
;;  Music & Fx Copyright (C) 2018 Alvaro Jover (@vorixo)
;;  This file is part of 77 ATTEMPTS: an Amstrad CPC Game made with CPCTelera
;;
;;  This program is free software: you can redistribute it and/or modify
;;  it under the terms of the GNU General Public License as published by
;;  the Free Software Foundation, either version 3 of the License, or
;;  (at your option) any later version.
;;
;;  This program is distributed in the hope that it will be useful,
;;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;  GNU General Public License for more details.
;;
;;  You should have received a copy of the GNU General Public License
;;  along with this program.  If not, see <http://www.gnu.org/licenses/>.
;;----------------------------------------------------------------------------------------------

.include "entities/entity.h.s"
.include "gamemanager/mapmanager.h.s"



;;=============================================
;; This function just call the function that
;;   check 
;; INPUTS:
;;      HL: XY coordinates (H = X; L = Y)
;;       A: tile ID
;; OUTPUTS:
;;       A: 0x00 if no colliding
;;          0x01 if colliding
;;       B: tile ID which entity is colliding
;;          If B == input A -> A = 0x01
;;          If B != input A -> A = 0x00
;; DESTROYS: AF, BC, DE, HL
;;=============================================
entityIsCollidingWithTileCall:

    push af                 ;; Save AF

    call mm_ConvertToTileCoordinates ;; HL = Converts absolute coordinates on tile local coordinates
    call mm_getTileIdByPosition      ;; Returns in A the tile ID
                                     ;; Destroys HL, AF, DE, BC

                            
    
    pop bc                  ;; Restore AF in BC

    cp b                    ;; Cmp A and B

    ld b, a                 ;; B = tile ID which entity is colliding

    jr z, colliding
        xor a            ;; Is not colliding, so A = 0
        ret

    colliding:
    
    ld a, #1                ;; Is colliding, so A = 1

    ret

;;=============================================
;; Function that checks if given entity
;;   is colliding with tile.
;; INPUTS:
;;      IY: entity
;;      HL: pointer to function to calculate
;;          first point
;;      DE: pointer to function to calculate
;;          second point
;;       A: tile ID
;;       B: offset
;; OUTPUTS:
;;       A: 0x00 if no colliding
;;          0x01 if colliding
;;      HL: tile IDs which entity is colliding
;;       H: First point to check
;;       L: Second point to check
;; DESTROYS: AF, BC, DE, HL
;;=============================================
entityIsCollidingWithTileGeneric::
    ld (first_function), hl             ;; first_function  = HL = First function to calculate first point
    ld (second_function), de            ;; second_function = DE = Second function to calculate second point

    ld de, #generic_selfmod_code        ;; | DE = generic_selfmod_code addr
    ld (de), a                          ;; | *DE = *(generic_selfmod_code) = A = tile ID (i.e. we save tile ID)
    inc de                              ;; | DE++ = *(generic_selfmod_code + 1) (i.e. next byte where we can load data)
    ld a, b                             ;; | A = B = offset value
    ld (de), a                          ;; | *DE = (*generic_selfmod_code + 1) = A = offset value
    dec de                              ;; | DE-- = *(generic_selfmod_code)
    ld a, (de)                          ;; | A = *DE = *(generic_selfmod_code) = tile ID
                                        ;;  \ Save tile ID and offset value and at the same time, swap values to avoid the use of stack
                                        ;;      for save temporally the value of offset

    jr generic_start

    generic_selfmod_code = .            ;; Self-modificable code memory begin
    .db 0x00                            ;; 1 byte reserved for tile ID
    .db 0x00                            ;; 1 byte reserved for offset value

    generic_start:

    first_function = . + 1
    call #0x00                          ;; Calling first function that will return 1 point

    ld de, #generic_selfmod_code        ;; DE = generic_selfmod_code addr
    ld a, (de)                          ;;  A = *DE = *(generic_selfmod_code) = tile ID

    call entityIsCollidingWithTileCall  ;; Calling function to check collision with the point calculated before

    ;; Function does not end here (if were possible) because we need to know both ID tiles which entity is colliding

    push af                             ;; We save A (i.e. 0x00 if previous call have not found collision and 0x01 if did)
    push bc                             ;; We save B (i.e. tile ID which entity is colliding and can be the same or different that input A)

    ld de, #generic_selfmod_code        ;; DE = generic_selfmod_code addr
    inc de                              ;; DE++ = generic_selfmod_code + 1 addr
    ld a, (de)                          ;;  A = *DE = *(generic_selfmod_code + 1) = offset value
    ld b, a                             ;;  B = A = offset value
    dec de                              ;; DE-- = generic_selfmod_code addr = tile ID

    second_function = . + 1
    call #0x00                          ;; Calling first function that will return 1 point

    ld a, (de)                          ;;  A = *DE = *(generic_selfmod_code + 1) = tile ID
                                        ;; Making this instruction after call because that functions
                                        ;;   that returns points destroys A register but no DE
                                        ;;   and for that reason it is made 'dec de' before the call.


    call entityIsCollidingWithTileCall  ;; Calling function to check collision with the point calculated before

    pop hl                              ;; HL = tile ID which entity collided before (H) & trash (L)
    ld l, b                             ;;  L = tile ID which entity have collided now

    cp #1
    jr nz, generic_end
        pop de                          ;; pop needed, if not, will fail due to how call/ret works (pop in DE for no reason)
        ret

    generic_end:                        ;; Last collision check did not collided with the tile ID that was our function input (A)

    pop af                              ;; A = collision result in first call to collision function check

    ret

;;=============================================
;; Function that checks if given entity
;;   is colliding with tile in his upper part.
;; INPUTS:
;;      IY: entity
;;       A: tile ID
;;       B: offset
;; OUTPUTS:
;;       A: 0x00 if no colliding
;;          0x01 if colliding
;;      HL: tile IDs which entity is colliding
;;       H: First point to check
;;       L: Second point to check
;; DESTROYS: AF, BC, DE, HL
;;=============================================
entityIsCollidingWithTileUpper::

    ld hl, #getUpLeftCoordinateUpperOffset      ;; HL = Up-Left point with  | Upper | offset
    ld de, #getUpRightCoordinateUpperOffset     ;; DE = Up-Right point with | Upper | offset

    jr entityIsCollidingWithTileGeneric         ;; A = 1 if collision | A = 0 if not collision

;;=============================================
;; Function that checks if given entity
;;   is colliding with tile in his bottom part.
;; INPUTS:
;;      IY: entity
;;       A: tile ID
;;       B: offset
;; OUTPUTS:
;;       A: 0x00 if no colliding
;;          0x01 if colliding
;;      HL: tile IDs which entity is colliding
;;       H: First point to check
;;       L: Second point to check
;; DESTROYS: AF, BC, DE, HL
;;=============================================
entityIsCollidingWithTileBottom::
    ld hl, #getDownLeftCoordinateBottomOffset   ;; HL = Down-Left point with  | Bottom | offset
    ld de, #getDownRightCoordinateBottomOffset  ;; DE = Down-Right point with | Bottom | offset

    jr entityIsCollidingWithTileGeneric         ;; A = 1 if collision | A = 0 if not collision

;;=============================================
;; Function that checks if given entity
;;   is colliding with tile in his left part.
;; INPUTS:
;;      IY: entity
;;       A: tile ID
;;       B: offset
;; OUTPUTS:
;;       A: 0x00 if no colliding
;;          0x01 if colliding
;;      HL: tile IDs which entity is colliding
;;       H: First point to check
;;       L: Second point to check
;; DESTROYS: AF, BC, DE, HL
;;=============================================
entityIsCollidingWithTileLeft::
    ld hl, #getUpLeftCoordinateLeftOffset       ;; HL = Up-Left point with   | Left | offset
    ld de, #getDownLeftCoordinateLeftOffset     ;; DE = Down-Left point with | Left | offset

    jr entityIsCollidingWithTileGeneric         ;; A = 1 if collision | A = 0 if not collision

;;=============================================
;; Function that checks if given entity
;;   is colliding with tile in his right part.
;; INPUTS:
;;      IY: entity
;;       A: tile ID
;;       B: offset
;; OUTPUTS:
;;       A: 0x00 if no colliding
;;          0x01 if colliding
;;      HL: tile IDs which entity is colliding
;;       H: First point to check
;;       L: Second point to check
;; DESTROYS: AF, BC, DE, HL
;;=============================================
entityIsCollidingWithTileRight::
    ld hl, #getUpRightCoordinateRightOffset     ;; HL = Up-Right point with   | Right | offset
    ld de, #getDownRightCoordinateRightOffset   ;; DE = Down-Right point with | Right | offset

    jr entityIsCollidingWithTileGeneric         ;; A = 1 if collision | A = 0 if not collision

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |*             |
;;  |              |
;;  |              |
;;  ----------------
;;
;;  Offset 2:
;;   *
;;
;;  ----------------
;;  |              |
;;  |              |
;;  |              |
;;  ----------------
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getUpLeftCoordinateUpperOffset:
    ld h, Ent_x(iy)             ;; H = X coordinate Up-Left
    ld a, Ent_y(iy)             ;; A = Y coordinate
    sub a, b                    ;; A -= offset
    ld l, a                     ;; L = A = Y coordinate Up-Left with offset

    ret

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |             *|
;;  |              |
;;  |              |
;;  ----------------
;;
;;  Offset 2:
;;                *
;;
;;  ----------------
;;  |              |
;;  |              |
;;  |              |
;;  ----------------
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getUpRightCoordinateUpperOffset:
    ld a, Ent_x(iy)             ;; A = X coordinate
    add a, Ent_w(iy)            ;; A += width
    dec a                       ;; A--
    ld h, a                     ;; H = A = X coordinate Up-Right
    ld a, Ent_y(iy)             ;; A = Y coordinate
    sub a, b                    ;; A -= offset
    ld l, a                     ;; L = A = Y coordinate Up-Right with offset

    ret

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |*             |
;;  |              |
;;  |              |
;;  ----------------
;;
;;  Offset 2:
;;     ----------------
;;  *  |              |
;;     |              |
;;     |              |
;;     ----------------
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getUpLeftCoordinateLeftOffset:
    ld a, Ent_x(iy)             ;; A = X coordinate
    sub a, b                    ;; A -= offset
    ld h, a                     ;; H = A = X coordinate Up-Left with offset
    ld l, Ent_y(iy)             ;; L = Y coordinate Up-Left

    ret

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |             *|
;;  |              |
;;  |              |
;;  ----------------
;;
;;  Offset 2:
;;  ----------------
;;  |              |  *
;;  |              |
;;  |              |
;;  ----------------
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getUpRightCoordinateRightOffset:
    ld a, Ent_x(iy)             ;; A = X coordinate
    add a, Ent_w(iy)            ;; A += width
    dec a                       ;; A--
    add a, b                    ;; A += offset
    ld h, a                     ;; H = A = X coordinate Up-Right with offset
    ld l, Ent_y(iy)             ;; L = Y coordinate Up-Right

    ret

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |              |
;;  |              |
;;  |*             |
;;  ----------------
;;
;;  Offset 2:
;;  ----------------
;;  |              |
;;  |              |
;;  |              |
;;  ----------------
;;
;;   *
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getDownLeftCoordinateBottomOffset:
    ld h, Ent_x(iy)             ;; H = X coordinate Down-Left
    ld a, Ent_y(iy)             ;; A = Y coordinate
    add a, Ent_h(iy)            ;; A += height
    dec a                       ;; A--
    add a, b                    ;; A += offset
    ld l, a                     ;; L = A = Y coordinate Down-Left with offset

    ret

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |              |
;;  |              |
;;  |             *|
;;  ----------------
;;
;;  Offset 2:
;;  ----------------
;;  |              |
;;  |              |
;;  |              |
;;  ----------------
;;
;;                *
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getDownRightCoordinateBottomOffset:
    ld a, Ent_x(iy)             ;; A = X coordinate
    add a, Ent_w(iy)            ;; A += width
    dec a                       ;; A--
    ld h, a                     ;; H = A = X coordinate Down-Right
    ld a, Ent_y(iy)             ;; A = Y coordinate
    add a, Ent_h(iy)            ;; A += height
    dec a                       ;; A--
    add a, b                    ;; A += offset
    ld l, a                     ;; L = A = Y coordinate Down-Right with offset

    ret

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |              |
;;  |              |
;;  |*             |
;;  ----------------
;;
;;  Offset 2:
;;     ----------------
;;     |              |
;;     |              |
;;  *  |              |
;;     ----------------
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getDownLeftCoordinateLeftOffset:
    ld a, Ent_x(iy)             ;; A = X coordinate
    sub a, b                    ;; A -= offset
    ld h, a                     ;; H = A = X coordinate Down-Left with offset
    ld a, Ent_y(iy)             ;; A = Y coordinate
    add a, Ent_h(iy)            ;; A += height
    dec a                       ;; A--
    ld l, a                     ;; L = A = Y coordinate Down-Left

    ret

;;=============================================
;;  It returns a point of an entity:
;;
;;  Offset 0:
;;  ----------------
;;  |              |
;;  |              |
;;  |             *|
;;  ----------------
;;
;;  Offset 2:
;;  ----------------
;;  |              |
;;  |              |
;;  |              |  *
;;  ----------------
;;
;;  INPUTS:
;;      IY: entity
;;       B: offset
;;  OUTPUTS:
;;       H: X coordinate of point
;;       L: Y coordinate of point
;;  DESTROYS: AF, HL
;;=============================================
getDownRightCoordinateRightOffset:
    ld a, Ent_x(iy)             ;; A = X coordinate
    add a, Ent_w(iy)            ;; A += width
    dec a                       ;; A--
    add a, b                    ;; A += offset
    ld h, a                     ;; H = A = X coordinate Down-Right with offset
    ld a, Ent_y(iy)             ;; A = Y coordinate
    add a, Ent_h(iy)            ;; A += height
    dec a                       ;; A--
    ld l, a                     ;; L = A = Y coordinate Down-Right

    ret

;;=============================================
;; This function checks if an entity is 
;;   colliding with a tile (box approach)
;; It checks 8 points from the corners and
;;   adds the offset to that points:
;;
;; Offset 0 (e.g. check if a boss have hit us):
;;  ---------------
;;  |*           *|
;;  |             |
;;  |*           *|
;;  ---------------
;; Offset 1 (e.g. check if hero is in the floor):
;;   *           *
;;  ---------------
;; *|             |*
;;  |             |
;; *|             |*
;;  ---------------
;;   *           *
;;
;; Offset 2 (e.g. special floor where we can levitate):
;;   *           *
;;
;;   ---------------
;; * |             | *
;;   |             |
;; * |             | *
;;   ---------------
;;
;;    *           *
;;
;; Offset 10 (e.g. special floor where we can fly)
;;
;; Offset -1 (e.g. special collisions checks (can be used for big sprites)):
;;  ---------------
;;  | *         * |
;;  |*           *|
;;  |             |
;;  |*           *|
;;  | *         * |
;;  ---------------
;;
;; INPUTS:
;;      IY: entity
;;       A: tile ID
;;       B: offset
;; OUTPUTS:
;;       A: 0x00 if no colliding
;;          0x01 if colliding
;; DESTROYS: AF, BC, DE, HL
;;=============================================

_0_col_tile = .         ;; Self-modified code addr begin
.db 0x00                ;; 1 byte reserved
.db 0x00                ;; 1 byte reserved
                        ;;  \ 2 bytes reserved

entityIsCollidingWithTile::

    ld hl, #_0_col_tile     ;; | HL = _0_col_tile addr
    ld (hl), a              ;; | *(_0_col_tile) = A = tile ID (we save our tile ID (NOTE: tile ID will not change in the execution of this function))
    inc hl                  ;; | HL++ = _0_col_tile + 1 = next byte
    ld (hl), b              ;; | *(_0_col_tile + 1) = B = offset (we save our offset value (NOTE: the offset value will not change in the execution of this function))
                            ;;  \ Self-modified code
                            ;;  \ Using self-modified code
                            ;;     for optimization (we avoid a lot of stack use)
                            ;;     just for use only 2 bytes

    entColStart:

    ;; Upper side

    ld hl, #getUpLeftCoordinateUpperOffset      ;; HL = Up-Left point and Up offset
    ld de, #getUpRightCoordinateUpperOffset     ;; DE = Up-Right point and Up offset

    call entityIsCollidingWithTileGeneric       ;; A = 1 if collision with our ID tile
    cp #1
    ret z

    ;; Bottom side

    ld hl, #_0_col_tile                         ;; HL = _0_col_tile addr
    ld a, (hl)                                  ;; A = *HL = *(_0_col_tile) = ID tile
    inc hl                                      ;; HL++ = _0_col_tile + 1 = offset addr
    ld b, (hl)                                  ;; B = *HL = offset value

    ld hl, #getDownLeftCoordinateBottomOffset   ;; HL = Down-Left point and Bottom offset
    ld de, #getDownRightCoordinateBottomOffset  ;; DE = Down-Right point and Bottom offset

    call entityIsCollidingWithTileGeneric       ;; A = 1 if collision with our ID tile
    cp #1
    ret z

    ;; Left side

    ld hl, #_0_col_tile                         ;; HL = _0_col_tile addr
    ld a, (hl)                                  ;; A = *HL = *(_0_col_tile) = ID tile
    inc hl                                      ;; HL++ = _0_col_tile + 1 = offset addr
    ld b, (hl)                                  ;; B = *HL = offset value

    ld hl, #getUpLeftCoordinateLeftOffset       ;; HL = Up-Left point and Left offset
    ld de, #getDownLeftCoordinateLeftOffset     ;; DE = Down-Left point and Left offset

    call entityIsCollidingWithTileGeneric       ;; A = 1 if collision with our ID tile
    cp #1
    ret z

    ;; Right side

    ld hl, #_0_col_tile                         ;; HL = _0_col_tile addr
    ld a, (hl)                                  ;; A = *HL = *(_0_col_tile) = ID tile
    inc hl                                      ;; HL++ = _0_col_tile + 1 = offset addr
    ld b, (hl)                                  ;; B = *HL = offset value

    ld hl, #getUpRightCoordinateRightOffset     ;; HL = Up-Right point and Right offset
    ld de, #getDownRightCoordinateRightOffset   ;; DE = Down-Right point and Right offset

    call entityIsCollidingWithTileGeneric       ;; A = 1 if collision with our ID tile
    cp #1
    ret z

    ;ld a, #0                                    ;; No collision detected -> return A = 0
    xor a                                       ;; No collision detected -> return A = 0
    ret
